/* Abelbeck Aviation Checklist - main library script
Copyright 2024 Frank Abelbeck <frank@abelbeck.info>

This file is part of the Abelbeck Aviation Checklist (AAC) toolbox.

The AAC toolbox is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the AAC toolbox.  If not, see <http://www.gnu.org/licenses/>.
*/

const ERR_MALCONFIG = "Abelbeck Aviation Checklist application was not configured properly.";

/**
 * Modify the document's inner HTML to display an error page with the given
 * reason, error message and a link to the given root directory for raw access.
 * 
 * @param {String} strReason - Text describing the reason.
 * @param {String} strError - Text describing the error.
 * @param {String} strRootDir - Path to the root directory.
 */
function setErrorPage(strReason,strErr,strRootDir) {
	document.body.innerHTML = `<header>
	<h2>Abelbeck Aviation Checklist Error</h2>
</header>
<main>
	<p>${strReason}</p>
	<p>${strErr}</p>
	<p><a href="${strRootDir}">Follow this link to the static files.</a></p>
</main>
<footer>
	<p>Abelbeck Aviation Checklist &copy; 2024 <a href="mailto:Frank@Abelbeck.info">Frank Abelbeck</a></p>
</footer>`
}


/*
 * Access Wake Lock API and Page Visibility API
 *   https://developer.mozilla.org/en-US/docs/Web/API/Screen_Wake_Lock_API
 *   https://developer.mozilla.org/en-US/docs/Web/API/Page_Visibility_API
 */
class VisibilityWakeLock {
	constructor() {
		this.request = this.request.bind(this);
		this.release = this.release.bind(this);
		this.onRequest = () => {};
		document.addEventListener("visibilitychange", async () => {
			await this.request()
		});
	}
	
	async request() {
		let boolWLState = false;
		try {
			// try to get a new WakeLockSentinel, and point this function to the release event
			// (i.e. re-request wake lock when the user client releases it)
			this._wl = await navigator.wakeLock.request("screen");
			this._wl.addEventListener("release", this.request);
			boolWLState = true;
		} catch (err) {
			// possible cause: page not visible
		}
		// custom callback, used to inform user about wake lock status
		this.onRequest(boolWLState);
	}
	
	async release() {
		try {
			await this._wl.release();
		} catch (err) {
			// pass
		}
	}
}


/**
 * Prepare initialisation: extract metadata related to the Abelbeck Web
 * Framework and construct the main object.
 */
async function prepareInitialisation() {
	/*
	 * get a wake lock respecting the page's visibility
	 * show wake lock state by adjusting class list of the wake lock node
	 */
	let vwl = new VisibilityWakeLock();
	await vwl.request();
	
	// read all meta nodes with names starting with aac and write name-content pairs to a dictionary
	let mapMetadata = new Map();
	document.querySelectorAll("meta[name^='aac']").forEach( (node) => { mapMetadata.set(node.name.slice(3), node.content); } );
	
	// if RootDir meta data is not defined, assume current directory
	if (!mapMetadata.has("RootDir")) mapMetadata.set("RootDir",".");
	
	// if I18nDir meta data is not defined, assume i18n subdirectory
	if (!mapMetadata.has("I18nDir")) mapMetadata.set("I18nDir","i18n");
	
	if (!mapMetadata.has("Module")) mapMetadata.set("Module","main");
	
	switch (mapMetadata.get("Module")) {
		case "checklist":
			import("./libChecklist.js")
				.then( (mod) => { (new mod.ChecklistManager(mapMetadata)).run(); })
				.catch( (err) => { setErrorPage(ERR_MALCONFIG,err,mapMetadata.get("RootDir")); } );
			break;
		case "main":
		default:
			import("./libMain.js")
				.then( (mod) => { (new mod.MainMenu(mapMetadata)).run(); })
				.catch( (err) => { setErrorPage(ERR_MALCONFIG,err,mapMetadata.get("RootDir")); } );
	}
}

/*
 * script entry point: only initialise when content is fully loaded
 */
if (document.readyState === "loading") {
	// document is still loading, add a listener for initialisation
	window.addEventListener("DOMContentLoaded",prepareInitialisation);
} else {
	// document already loaded, DOMContentLoaded won't fire, so initialise explicitly
	prepareInitialisation();
}
