/* Abelbeck Aviation Checklist - web document manager library script
Copyright 2024 Frank Abelbeck <frank@abelbeck.info>

This file is part of the Abelbeck Aviation Checklist (AAC) toolbox.

The AAC toolbox is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the AAC toolbox.  If not, see <http://www.gnu.org/licenses/>.
*/


export class WebDocumentManager {
	
	INT_MAX_FILESIZE = 2**28 // 256*1024*1024, read functions will throw an error if a file-to-read is bigger (override in metadata, key "maxFileSize")
	
	/**
	 * Constructor: initialise this object, process configuration metadata.
	 * 
	 * @constructor
	 * 
	 * @param {Map} mapMeta - A map object with key-value pairs of configuration data
	 */
	constructor(mapMeta) {
		this.decoderUtf8 = new TextDecoder("utf-8");
		this.decoderAscii = new TextDecoder("ascii");
		
		this._urlState = new URL(document.location);
		this._strUrlPrev = this._urlState.href;
		this._strPathReferrer = undefined;
		
		this._mapMeta = mapMeta;
		
		// process metadata
		this._mapMeta.set("MaxFileSize", Number(this._mapMeta.get("MaxFileSize")) );
		if (isNaN(this._mapMeta.get("MaxFileSize"))) this._mapMeta.set("MaxFileSize", this.INT_MAX_FILESIZE);
		
		const methods = Object.getOwnPropertyNames(Object.getPrototypeOf(this));
		methods.filter((method) => (method !== "constructor")).forEach((method) => { this[method] = this[method].bind(this);});
	}
	
	static constructNode(strTagName,varContent,objAttributes) {
		let node = document.createElement(strTagName);
		if (typeof(varContent) == "string") {
			node.innerHTML = varContent;
		} else if (varContent !== undefined && varContent !== null) {
			try {
				/*
				 * try to iterate over varContent
				 */
				try {
					for (const nodeChild of varContent) {
						node.appendChild(nodeChild);
					}
				} catch (err) {
					console.warn(`constructNode(): ignoring child node ${nodeChild} (${err})`);
				}
			} catch (err) {
				try {
					/*
					* try to append varContent as a child
					*/
					node.appendChild(varContent)
				} catch (err) {
					console.warn(`constructNode(): ignoring child node ${varContent} (${err})`);
				}
			}
		}
		WebDocumentManager.addAttributes(node,objAttributes);
		return node;
	}
	
	static addAttributes(node,objAttributes) {
		try {
			for (const [key,value] of Object.entries(objAttributes)) {
				node.setAttribute(key,value);
			}
		} catch (err) {}
	}
	
	static constructText(strContent) {
		return document.createTextNode(strContent);
	}
	
	static constructButton(strContent,fnCallbackClick,objAttributes) {
		try {
			objAttributes.type = "button";
		} catch (err) {
			objAttributes = { type: "button" };
		}
		let node = this.constructNode("button",strContent,objAttributes);
		if (typeof(fnCallbackClick) == "function") {
			node.addEventListener("click",fnCallbackClick);
		}
		return node;
	}
	
	/**
	 * Start the URL state processing. Install a listener to popstate,
	 * call updateState(), and read i18n data, if available
	 * 
	 * This callback method calls the following generic methods:
	 * 
	 *  1. runEnter()
	 *  2. runMain()
	 *  3. runLeave()
	 * 
	 * @async
	 */
	async run() {
		await this.runEnter();
		await this.runMain();
		await this.runLeave();
	}
	
	/**
	 * This generic method is called first when calling run(). Its purpose is to
	 * set-up the run environment.
	 * 
	 * In its most basic form it adds a window event listener for the popstate
	 * event, and clears the body element.
	 * 
	 * You most likely want to re-implement this method, but don't forget to
	 * call this basic implementation (super.runEnter()).
	 * 
	 * @async
	 */
	async runEnter() {
		window.addEventListener("popstate", async (event) => { await this.updateState(); });
		document.body.replaceChildren();
	}
	
	/**
	 * This generic method is called after runEnter(). Its purpose is to
	 * further elaborate the run environment.
	 * 
	 * In its most basic form it does nothing.
	 * 
	 * You most likely want to re-implement this method.
	 * 
	 * @async
	 */
	async runMain() {
	}
	
	/**
	 * This generic method is called after runMain(). Its purpose is to clean-up
	 * the run environment.
	 * 
	 * In its most basic form it calls updateState().
	 * 
	 * You most likely want to re-implement this method, but don't forget to
	 * call this basic implementation (super.runLeave()).
	 * 
	 * @async
	 */
	async runLeave() {
		await this.updateState();
	}
	
	/**
	 * Update the document, based on the current URL. This callback method 
	 * is triggered by popstate events and calls the following generic methods:
	 * 
	 *  1. updateStateEnter()
	 *  2. updateStateMain()
	 *  3. updateStateLeave()
	 * 
	 * @async
	 */
	async updateState() {
		await this.updateStateEnter();
		await this.updateStateMain();
		await this.updateStateLeave();
	}
	
	/**
	 * This generic method is called first during state update. Its purpose is
	 * to set-up the environment for state processing.
	 * 
	 * In its most basic form it starts the throbber, generates the _urlState
	 * dictionary with parameters (including the fragment string addressed by
	 * '#'), and determines the referrer string.
	 * 
	 * You most likely want to re-implement this method, but don't forget to
	 * call this basic implementation (super.updateStateEnter()).
	 * 
	 * @async
	 */
	async updateStateEnter() {
		this.startThrobber();
		// save query params and fragment string of the document's URI to the state dictionary
		this._urlState = new URL(document.location);
		// determine the referrer, from query string over metadata definition to the document's referrer info
		this._strPathReferrer = this._urlState.searchParams.get("referrer");
		if (!this._strPathReferrer) {
			this._strPathReferrer = this._mapMeta.get("referrer");
			if (!this._strPathReferrer) {
				this._strPathReferrer = document.referrer;
				if (!this._strPathReferrer) self._strPathReferrer = undefined;
			}
		}
	}
	
	/**
	 * This generic method is used to carry out the main state update task.
	 * 
	 * In its most basic form it does nothing.
	 * 
	 * You most likely want to re-implement this method.
	 * 
	 * @async
	 */
	async updateStateMain() {
	}
	
	/**
	 * This generic method is used to clean up after the state update task.
	 * 
	 * In its most basic form it calls the following methods:
	 * 
	 *  1. processAnchors() - intercept all page-internal links
	 *  2. stopThrobber() - stop and remove throbber icon
	 * 
	 * You most likely want to re-implement this method, but don't forget to
	 * call this basic implementation (super.updateStateLeave()).
	 * 
	 * @async
	 */
	async updateStateLeave() {
		this.processAnchors();
		this.updateUrl();
		this.stopThrobber();
	}
	
	async readDirectoryIndex(strPath) {
		/*
		 * an index file lists filenames
		 * empty lines and lines starting with # are ignored
		 * returned is an array of filenames
		 */
		return (await this.readText(strPath)).replaceAll(/^\s*#.*$/mg,"").split("\n").map( (strLine) => strLine.trim()).filter( (strLine) => strLine.length > 0 )
	}
	
	getLanguage() {
		return document.querySelector("html").getAttribute("lang");
	}
	
	setLanguage(strLanguageCode) {
		document.querySelector("html").setAttribute("lang",strLanguageCode);
	}
	
	updateUrl(boolDoReplace=false) {
		if (boolDoReplace) {
			window.history.replaceState({},"",this._urlState);
		} else {
			window.history.pushState({},"",this._urlState);
		}
	}
	
	/**
	 * 
	 */
	interceptInternalAnchorClick(event,urlTarget) {
		event.preventDefault();
		window.history.pushState( {}, "", urlTarget);
		this.updateState();
	}
	
	/**
	 * 
	 */
	interceptExternalAnchorClick(event,urlTarget) {
		event.preventDefault();
		location.href = urlTarget.href;
	}
	
	/**
	 * 
	 */
	processAnchors() {
		for (const node of document.querySelectorAll("a[href]")) {
			let urlTarget = new URL(node.getAttribute("href"),document.location);
			if (urlTarget.origin == document.location.origin && urlTarget.pathname == document.location.pathname) {
				// if anchor points to a resource inside the current page (i.e. query, fragment),
				// install the intercept handler to prevent the browser from reloading
				node.addEventListener("click", (event) => { this.interceptInternalAnchorClick(event,urlTarget); } );
			} else {
				// anchor pointing outside this document
				node.addEventListener("click", (event) => { this.interceptExternalAnchorClick(event,urlTarget); } );
			}
		};
	}
	
	
	/**
	 * Filter a response: check if it's ok, if it has a body and if its length
	 * doesn't exceed the defined maxFileSize.
	 * 
	 * @async
	 * 
	 * @param {Response} response - A Response instance.
	 * 
	 * @return {Response} The input argument, unaltered.
	 * 
	 * @throws Will throw an Error if response is not OK, has no body or the
	 * content length exceeds the limit set in maxFileSize.
	 */
	filterResponse(response) {
		if (!response.ok) throw new Error(
			"fetch failed",
			{ cause:response.statusText }
		);
		if (!response.body) throw new Error("fetch returned empty body");
		
		const intLengthContent = response.headers.get("content-length");
		if (intLengthContent > this._mapMeta.get("MaxFileSize")) throw new Error(`content length ${intLengthContent} exceeds limit ${this._mapMeta.get("MaxFileSize")}`);
		
		return response;
	}
	
	/**
	 * Read contents of a zlib-compressed stream.
	 * (cf. https://developer.mozilla.org/en-US/docs/Web/API/Compression_Streams_API)
	 * 
	 * @async
	 * 
	 * @param {ReadableStream} streamCompressed
	 * 
	 * @return {Uint8Array} Content read from the stream; possibly empty in case
	 * of an error.
	 * 
	 * @throws Will throw an error if decompression failed.
	 */
	async deflateStream(streamCompressed) {
		return new Uint8Array( await new Response(streamCompressed.pipeThrough(new DecompressionStream("deflate"))).arrayBuffer() );
	}
	
	/**
	 * Read contents of a zlib-compressed array.
	 * 
	 * Calls deflateStream() on a ReadableStream composed from arrCompressed.
	 * 
	 * @async
	 * 
	 * @param {Array} arrCompressed
	 * 
	 * @return {Uint8Array} Content read from the array; possibly empty in
	 * case of an error.
	 * 
	 * @throws Will throw an error if decompression failed.
	 */
	async deflateArray(arrCompressed) {
		return await this.deflateStream( new Blob([arrCompressed]).stream() );
	}
	
	/**
	 * Read contents of a zlib-compressed ressource.
	 * 
	 * Fetches the resource, extracts the compressed stream, and calls
	 * deflateStream().
	 * 
	 * @async
	 * 
	 * @param {String} strPath - Path to the ressource.
	 * 
	 * @return {Uint8Array} Content read from the ressource; possibly empty in
	 * case of an error.
	 * 
	 * @throws Will throw an error if fetching the resource failed.
	 */
	async readZlib(strPath) {
		return await fetch(strPath, { cache:"default" })
			.then( (response) => this.filterResponse(response) )
			.then( (response) => response.body )
			.then( (compressedStream) => this.deflateStream(compressedStream) );
	}
	
	/**
	 * Read a UTF-8 encoded text ressource.
	 * 
	 * @async
	 * 
	 * @param {String} strPath - Path to the ressource.
	 * 
	 * @return {String} Content read from the ressource; possibly empty in case
	 * of an error.
	 * 
	 * @throws Will throw an error if fetching the resource failed.
	 */
	async readText(strPath) {
		return await fetch(strPath, { cache:"default" })
			.then( (response) => this.filterResponse(response) )
			.then( (response) => response.text() );
	}
	
	/**
	 * Read an HTML ressource.
	 * 
	 * @async
	 * 
	 * @param {String} strPath - Path to the ressource.
	 * 
	 * @return {HTMLDocument} Content read from the ressource; possibly empty in
	 * case of an error.
	 * 
	 * @throws Will throw an error if fetching the resource failed.
	 */
	async readHtml(strPath) {
		let parser = new DOMParser();
		return parser.parseFromString(await this.readText(strPath), "text/html");
	}
	
	/**
	 * Read a JSON ressource.
	 * 
	 * @async
	 * 
	 * @param {String} strPath - Path to the ressource.
	 * 
	 * @return {String} Content read from the ressource; possibly empty in case
	 * of an error.
	 * 
	 * @throws Will throw an error if fetching the resource failed.
	 */
	async readJson(strPath) {
		return await fetch(strPath, { cache:"default" })
			.then( (response) => this.filterResponse(response) )
			.then( (response) => response.json() );
	}
	
	/**
	 * Read a binary ressource.
	 * 
	 * @async
	 * 
	 * @param {String} strPath - Path to the ressource.
	 * 
	 * @return {Uint8Array} Content read from the ressource; possibly empty in
	 * case of an error.
	 * 
	 * @throws Will throw an error if fetching the resource failed.
	 */
	async readBinary(strPath) {
		return await fetch(strPath, { cache:"default" })
			.then( (response) => this.filterResponse(response) )
			.then( (response) => response.arrayBuffer() )
			.then( (arrBuff) => new Uint8Array(arrBuff) );
	}
	
	/**
	 * Read and parse a directory listing.
	 * 
	 * @async
	 * 
	 * @param {String} strPath - Path to the ressource.
	 * 
	 * @return {Array} A 3-tuple (strDir,arrDirs,arrFiles) with the current
	 * directory name, an array of subdirectory names and an array of file
	 * names. Might be empty in case of an error.
	 * 
	 * @throws Will throw an error if fetching the resource failed.
	 */
	async readDirectory(strPath) {
		let arrMatchAll = await fetch(strPath, { cache:"default" })
			.then( (response) => this.filterResponse(response) )
			.then( (response) => response.text() )
			.then( (strText) => strText.matchAll(/<a.*?href=["']((?!\.\.\/|\/|\.\.|#.*?|\?.*?).*?)["']/gs) );
			// .catch( (error) => {
				// console.debug(error);
				// return [];
			// })
		let arrDirs = [];
		let arrFiles = [];
		for (const arrMatch of arrMatchAll) {
			if (arrMatch[1].endsWith("/")) {
				arrDirs.push(arrMatch[1]);
			} else {
				arrFiles.push(arrMatch[1]);
			}
		}
		return [strPath,arrDirs,arrFiles];
	}
	
	/**
	 * Translate a Uint8Array to a String by assuming UTF-8 encoding.
	 * 
	 * @param {Uint8Array} arrValue
	 * @return {String} Decoded UTF-8 string.
	 */
	decodeUtf8(arrValue) {
		return this.decoderUtf8.decode(arrValue);
	}
	
	/**
	 * Translate a Uint8Array to a String by assuming ASCII encoding.
	 * 
	 * @param {Uint8Array} arrValue
	 * @return {String} Decoded ASCII string.
	 */
	decodeAscii(arrValue) {
		return this.decoderAscii.decode(arrValue);
	}
	
	/**
	 * Convert a string to a Uint8Array.
	 * 
	 * @param {String} strValue
	 * @return {Uint8Array} Array of byte values of the encoded string.
	 */
	static convertStringToByteArray(strValue) {
		return Uint8Array.from(strValue,(c) => c.charCodeAt(0));
	}
	
	static convertByteArrayToString(uint8arrBytes) {
		return String.fromCharCode(...uint8arrBytes);
	}
	
	static castArrayToUintBE(arrInput) {
		return arrInput.reduce( (acc,val) => (acc << 8) + val );
	}
	
	static areArraysEqual(arrA,arrB) {
		if (arrA.length != arrB.length)
			return false;
		
		for (let i = 0; i < arrA.length; ++i)
			if (arrA[i] != arrB[i])
				return false;
		
		return true;
	}
	
	static castArrayToHex(arrInput) {
		return arrInput.reduce((acc,val) => acc + val.toString(16).padStart(2,"0"),"");
	}
	
	startThrobber() {
		// insert a loader div
		try {
			let nodeMain = document.querySelector("main");
			let nodeSpinner = document.createElement("div");
			nodeSpinner.classList.add("spinWheel");
			nodeMain.insertBefore(nodeSpinner,nodeMain.firstChild);
		} catch (err) {}
	}
	
	stopThrobber() {
		try {
			document.querySelector(".spinWheel").remove();
		} catch (err) {}
	}
	
	async readHeader(strPath) {
		return await fetch(strPath, { method: "HEAD", cache:"default" })
			.then( (response) => {
				if (!response.ok) throw new Error(
					"fetch failed",
					{
						cause:response.statusText,
					}
				);
				return response.headers;
			});
	}
	
	/*
	// JSON --> stream
	const stream = new Blob([JSON.stringify(data)], { type: 'application/json' }).stream();
	
	// stream --> gzip stream
	const compressedReadableStream = stream.pipeThrough( new CompressionStream("gzip") );
	
	// gzip stream --> response
	const compressedResponse = await new Response(compressedReadableStream);
	
	const blob = await compressedResponse.blob();
	
	// Get the ArrayBuffer
	const buffer = await blob.arrayBuffer();
	
	// convert ArrayBuffer to base64 encoded string
	const compressedBase64 = btoa(String.fromCharCode(...new Uint8Array(buffer)));
	
	
	// Get response Blob
	const blob = await compressedResponse.blob();
	*/
	
	async waitForEvent(node, strType) {
		return new Promise( (resolve) => {
			node.addEventListener(strType, resolve, { once:true });
		});
	}
	
	async loadFile() {
		/*
		 * 1) create a hidden input element
		 * 2) create a promise to wait for the next change event
		 * 3) add input element to document, emulate a click, wait for promise and remove node again
		 * 4) extract filename from files attribute
		 * 5) create and return a new promise, waiting for the load event of a FileReader
		 * 
		 * usage: await this.loadFile()
		 * returns a blob;
		 */
		let objData, promise, file;
		let node = WebDocumentManager.constructNode("input","",{
			type: "file",
			hidden: "hidden",
		});
		promise = this.waitForEvent(node, "change");
		document.body.appendChild(node);
		node.click();
		await promise;
		document.body.removeChild(node);
		
		return new Promise( (resolve, reject) => {
			let contents = "";
			const reader = new FileReader();
			reader.addEventListener("load", async (event) => {
				resolve( new Blob( [event.target.result], { type: node.files[0].type }) );
			});
			reader.addEventListener("error", async (event) => {
				reject(event);
			});
			reader.readAsArrayBuffer(node.files[0]);
		});
	}
	
	saveFile(strFilenameDefault,blob) {
		/*
		 * save a blob to a file via hidden anchor pointing to a data URL
		 */
		let strObjUrl = URL.createObjectURL( blob );
		let node = WebDocumentManager.constructNode("a","",{
			download: strFilenameDefault,
			href: strObjUrl
		});
		document.body.appendChild(node);
		node.click();
		document.body.removeChild(node);
		URL.revokeObjectURL(strObjUrl);
	}
	
	convertObjToJsonBlob(objData) {
		return new Blob( [JSON.stringify(objData)], { type: "application/json" } );
	}
	
	async convertJsonBlobToObj(blob) {
		return JSON.parse( await blob.text() ); 
	}
}


