/* Abelbeck Aviation Checklist - variable management library script
Copyright 2024 Frank Abelbeck <frank@abelbeck.info>

This file is part of the Abelbeck Aviation Checklist (AAC) toolbox.

The AAC toolbox is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the AAC toolbox.  If not, see <http://www.gnu.org/licenses/>.
*/

import {
	AsyncIOLock,
	AsyncIOMap,
	AsyncIOSet
} from "./libAio.js";


import {
	StackMachine
} from "./libStackMachine.js";


export class Variable {
	
	constructor() {
		this._boolEmpty = true;
		this._value = null;
		this._valueInitial = null;
		this._lock = new AsyncIOLock();
		this._strDescription = "";
		this._strFormat = "";
		this._numFormatWidth = undefined;
		this._numFormatPrecision = undefined;
		this._strFormatSign = "-";
		this._strFormatPad = " ";
		this._strFormatType = "s";
		this._boolFormatSuppress = true;
		this._strPrefix = "";
		this._strSuffix = "";
		
		const methods = Object.getOwnPropertyNames(Object.getPrototypeOf(this));
		methods.filter((method) => (method !== "constructor")).forEach((method) => { this[method] = this[method].bind(this);});
	}
	
	convert(valNew) {
		this._boolEmpty = (valNew === null || valNew === undefined);
		return valNew;
	}
	
	isValid(valNew) {
		return true;
	}
	
	async reset() {
		await this._lock.write( async () => {
			this._value = this._valueInitial;
		});
	}
	
	async setFormat(strFormat) {
		/*
		 * parse a format string as a subset of Python's Format Specification Mini-Language
		 * 
		 * format_spec     ::=  [sign]["0"][width]["." precision][type]["!"]
		 * sign            ::=  "+" | "-" | " "
		 * width           ::=  digit+
		 * precision       ::=  digit+
		 * type            ::=  "b" | "d" | "f" | "o" | "s" | "x" | "X"
		 * 
		 * /(.*?)?{([+ -])?(0)?([0-9]+)?(?:.([0-9]+))?([bdfosxX])?(!)?}(.*)?/ -> [match,prefix,sign,zero,width,grouping,precision,type,suppress,suffix]
		 * 
		 * sign: only for numbers
		 * sign "+": always show a sign symbol
		 * sign "-": only show a minus sign symbol
		 * sign " ": show a space for plus, and minus for negative values
		 * 
		 * "0": left-pad numbers after the sign before the digits
		 * 
		 * width: minimum format string length, including sign, decimal point
		 * precision: maximum format string length (strings), fraction precision (f type)
		 * 
		 * type: convert to string (s), binary integer (b), fixed-precision number (f),
		 *       octal integer (o), hexadecimal integer (x; X for upper case A-F), or decimal integer (d)
		 * 
		 * "!": if number conversion results in NaN, Infinity or -Infinity, output is suppressed;
		 *      with "!" also non-finite values are shown.
		 */
		
		await this._lock.write( async () => {
			try {
				this._strFormat = strFormat;
				let [strMatch,strPrefix,strSign,strPad,strWidth,strPrecision,strType,strSuppress,strSuffix] = strFormat.match(
					/(.*?)?{([+ -])?(0)?([0-9]+)?(?:.([0-9]+))?([bdfosxX])?(!)?}(.*)?/
				);
				this._numFormatWidth = (strWidth === undefined) ? undefined : parseInt(strWidth);
				this._numFormatPrecision = (strPrecision === undefined) ? undefined : parseInt(strPrecision);
				this._strFormatSign = (strSign === undefined) ? "-" : strSign;
				this._strFormatPad = (strPad === undefined) ? " " : "0" ;
				this._strFormatType = (strType === undefined) ? "s" : strType;
				this._boolFormatSuppress = (strSuppress !== "!");
				this._strPrefix = (strPrefix === undefined) ? "" : strPrefix;
				this._strSuffix = (strSuffix === undefined) ? "" : strSuffix;
			} catch (err) {}
		});
	}
	
	async getFormat() {
		let value;
		await this._lock.read( async () => {
			value = this._strFormat;
		});
		return value;
	}
	
	async getPrefixSuffix() {
		let value = ["",""];
		await this._lock.read( async () => {
			value = [this._strPrefix,this._strSuffix];
		});
		return value;
	}
	
	async setDescription(strDescription) {
		await this._lock.write( async () => {
			this._strDescription = String(strDescription);
		});
	}
	
	async getDescription() {
		let value;
		await this._lock.read( async () => {
			value = this._strDescription;
		});
		return value;
	}
	
	async getValue() {
		let value;
		await this._lock.read( async () => {
			value = this._value;
		});
		return value;
	}
	
	async getInitialValue() {
		let value;
		await this._lock.read( async () => {
			value = this._valueInitial;
		});
		return value;
	}
	
	async isEmpty() {
		let boolEmpty = false;
		await this._lock.read( async () => {
			boolEmpty = this._boolEmpty;
		});
		return boolEmpty;
	}
	
	async setInitialValue(valNew) {
		await this._lock.write( async () => {
			this._valueInitial = this.convert(valNew);
		});
	}
	
	async setValue(valNew) {
		let boolHasChanged = false;
		valNew = this.convert(valNew);
		if (this.isValid(valNew)) {
			await this._lock.write( async () => {
				boolHasChanged = valNew !== this._value;
				this._value = valNew;
			});
		}
		return boolHasChanged;
	}
	
	format(value) {
		if (this._strFormatType === "s") {
			/*
			 * format as string: cast to string, pad with padding char to reach width, slice to cut at precision
			 */
			value = this._strPrefix + String(value).padStart(this._numFormatWidth,this._strFormatPad).slice(0,this._numFormatPrecision) + this._strSuffix;
		} else {
			/*
			 * format as number: cast to number, retrieve sign
			 */
			value = Number(value);
			if (this._boolFormatSuppress && !isFinite(value)) {
				value = "";
			} else {
				let numSign = Math.sign(value);
				let numVal  = Math.abs(value);
				let strSign = "";
				if (this._strFormatSign === "+") {
					if (numSign > 0) {
						strSign = "+";
					} else if (numSign <= 0) {
						strSign = "-";
					}
				} else if (this._strFormatSign === " ") {
					if (numSign > 0 || numSign === 0) {
						strSign = " ";
					} else {
						strSign = "-";
					}
				}
				let numWidth = Math.max(this._numFormatWidth - strSign.length,0);
				
				/*
				* process value, depending on type; finally pad to width and add sign char
				* but: if ! was appended, and value is NaN, +Inf or -Inf, return empty string
				*/
				switch (this._strFormatType) {
					case "f":
						if (this._numFormatPrecision !== undefined) {
							value = numVal.toFixed(this._numFormatPrecision);
						} else {
							value = String(value);
						}
						break;
					case "b":
						value = Math.round(numVal).toString(2);
						break;
					case "d":
						value = Math.round(numVal).toString(10);
						break;
					case "o":
						value = Math.round(numVal).toString(8);
						break;
					case "x":
						value = Math.round(numVal).toString(16).toLowerCase();
						break;
					case "X":
						value = Math.round(numVal).toString(16).toUpperCase();
						break;
				}
				value = this._strPrefix + strSign + value.padStart(this._numFormatWidth,this._strFormatPad) + this._strSuffix;
			}
		}
		return value;
	}
	
	async toString() {
		return this.format(await this.getValue());
	}
	
	static idiv(dividend,divisor) {
		return Math.floor(dividend / divisor);
	}
	
	static divmod(dividend,divisor) {
		let quotient = Math.floor(dividend/divisor);
		return [quotient,dividend - divisor*quotient];
	}
	
	static modulo(dividend,divisor) {
		return dividend - divisor*Math.floor(dividend/divisor);
	}
}


export class VariableString extends Variable {
	constructor(valInit) {
		super();
		this._value = this.convert(valInit);
		this._valueInitial = this._value;
	}
	
	convert(valNew) {
		this._boolEmpty = (valNew === "" || valNew === undefined || valNew === null);
		return String(valNew);
	}
}


export class VariableRegExp extends VariableString {
	constructor(valInit,reCheck) {
		super();
		try {
			this._reCheck = new RegExp(reCheck);
		} catch (err) {
			this._reCheck = /.*/;
		}
		if (valInit === undefined) valInit = "";
		this._value = valInit;
		this._valueInitial = valInit;
	}
	
	isValid(strNew) {
		try {
			return this._reCheck.test(strNew);
		} catch (err) {}
		return false;
	}
}


export class VariableNumber extends Variable {
	constructor(valInit) {
		super();
		this._value = this.convert(valInit);
		this._valueInitial = this._value;
		this._numFormatWidth = undefined;
		this._numFormatPrecision = undefined;
		this._strFormatSign = "-";
		this._strFormatPad = " ";
		this._strFormatType = "f";
		this._boolFormatSuppress = true;
		this._strPrefix = "";
		this._strSuffix = "";
	}
	
	convert(valNew) {
		this._boolEmpty = (valNew === "" || valNew === undefined || valNew === null);
		if (!this._boolEmpty) {
			valNew = Number(valNew); // if Number can't parse this, NaN is returned
		} else {
			valNew = NaN;
		}
		return valNew;
	}
}


export class VariableOperatingHours extends Variable {
	constructor(valInit) {
		super(valInit);
		this._value = this.convert(valInit);
		this._valueInitial = this._value;
		this._strUnit = " h";
	}
	
	convert(valNew) {
		this._boolEmpty = (valNew === "" || valNew === undefined || valNew === null);
		if (this._boolEmpty) {
			valNew = NaN;
		} else {
			try {
				let [strAll,strHours,strMinutes] = valNew.match(/^([+-]?(?:[0-9]+))(?::([0-5]?[0-9]))?$/);
				let intHours = parseInt(strHours);
				let intMinutes = parseInt(strMinutes);
				if (isNaN(intMinutes)) {
					valNew = intHours;
				} else {
					valNew = intHours * 60 + intMinutes;
				}
			} catch (err) {
				valNew = super.convert(valNew);
			}
		}
		return valNew;
	}
	
	
	async toString() {
		let value = await this.getValue();
		if (!isNaN(value)) {
			let [intHours, intMinutes] = Variable.divmod(value, 60);
			if (intMinutes < 10) {
				value = `${intHours}:0${intMinutes}${this._strUnit}`;
			} else {
				value = `${intHours}:${intMinutes}${this._strUnit}`;
			}
		} else {
			value = "";
		}
		return value;
	}
}


export class VariableTime extends VariableOperatingHours {
	constructor(valInit) {
		super(valInit);
		this._strUnit = " Z";
	}
	
	convert(valNew) {
		return Variable.modulo(super.convert(valNew),1440);
	}
}


export class VariableManager {
	
	static mapTypes = new Map([
		["string",         VariableString],
		["number",         VariableNumber],
		["opHrs",          VariableOperatingHours],
		["time",           VariableTime],
	]);
	
	constructor() {
		this._mapVars = new AsyncIOMap(); // map name to variable
		this._mapSubscribers = new AsyncIOMap(); // map name to set of callback functions
		this._stackMachine = new StackMachine();
		
		const methods = Object.getOwnPropertyNames(Object.getPrototypeOf(this));
		methods.filter((method) => (method !== "constructor")).forEach((method) => { this[method] = this[method].bind(this);});
	}
	
	async clear() {
		await this._mapVars.clear();
		await this._mapSubscribers.clear();
		await this._stackMachine.clear();
	}
	
	async add(strName,strType,strDescription,strFormat,strValue,strExpression) {
		let clsNew, valNew;
		if (!this._stackMachine.isIdentifier(strName)) {
			console.error(`invalid variable identifier '${strName}'`);
			return;
		}
		try {
			clsNew = VariableManager.mapTypes.get(strType);
		} catch (err) {
			console.warn(`not adding variable '${strName}': type '${strType}' unknown (${err})`);
			return;
		}
		try {
			valNew = new clsNew();
			await this._mapVars.set(strName,valNew);
			if (strDescription !== undefined) {
				await valNew.setDescription(strDescription);
			}
			if (strFormat !== undefined) {
				await valNew.setFormat(strFormat);
			}
		} catch (err) {
			console.warn(`not adding variable '${strName}': init of type '${strType}' with value '${strValue}' failed (${err})`);
			return;
		}
	}
	
	async setExpression(strName, strExpression) {
		// console.log("setExpression " + strName + " " + strExpression);
		if (strExpression !== undefined) {
			let variable = await this._mapVars.get(strName);
			// console.log(variable);
			if (variable !== undefined) {
				let setDependencies = await this._stackMachine.register(strExpression);
				// console.log(`adding ${strName} = '${strExpression}' with dependencies ${setDependencies}`);
				for (const strDependency of setDependencies) {
					if (strDependency !== strName) {
						await this.subscribe(strDependency, async (variable) => {
							try {
								// console.log("evaluating " + strName + " = " + strExpression);
								await this.setValue(strName, await this.execute(strExpression));
							} catch (err) {
								console.warn(`failed to update auto-expression '${strName}'='${strExpression}': ${err}`);
							}
						});
					}
				}
			}
		}
	}
	
	async execute(strExpression) {
		try {
			return await this._stackMachine.execute(strExpression,this);
		} catch (err) {
			console.warn(`failed to execute '${strExpression}': ${err}`);
			return undefined;
		}
	}
	
	async has(strName) {
		return await this._mapVars.has(strName);
	}
	
	async getValue(strName) {
		try {
			return await (await this._mapVars.get(strName)).getValue();
		} catch (err) {
			console.warn(`failed to get value for '${strName}': ${err}`);
			return undefined;
		}
	}
	
	async getInitialValue(strName) {
		try {
			return await (await this._mapVars.get(strName)).getInitialValue();
		} catch (err) {
			console.warn(`failed to get initial value for '${strName}': ${err}`);
			return undefined;
		}
	}
	
	async getString(strName) {
		try {
			return await (await this._mapVars.get(strName)).toString();
		} catch (err) {
			console.warn(`failed to get string for '${strName}': ${err}`);
			return undefined;
		}
	}
	
	async getDescription(strName) {
		try {
			return (await this._mapVars.get(strName)).getDescription();
		} catch (err) {
			console.warn(`failed to get description for '${strName}' : ${err}`);
			return undefined;
		}
	}
	
	async isEmpty(strName) {
		try {
			return (await this._mapVars.get(strName)).isEmpty();
		} catch (err) {
			console.warn(`failed to get empty state for '${strName}' : ${err}`);
			return false;
		}
	}
	
	async setValue(strName,valNew) {
		// console.log("setValue " + strName + " " + valNew);
		let boolHasChanged = false;
		let variable = await this._mapVars.get(strName);
		if (variable) {
			boolHasChanged = await variable.setValue(valNew);
		}
		if (boolHasChanged) {
			await this.callSubscribers(strName);
		}
		return boolHasChanged;
	}
	
	async setInitialValue(strName,valNew) {
		let variable = await this._mapVars.get(strName);
		if (variable) {
			await variable.setInitialValue(valNew);
		}
	}
	
	async reset() {
		await this._mapVars.forEach( async (strName,variable) => {
			await variable.reset();
			await this.callSubscribers(strName);
		});
	}
	
	async subscribe(strName,fnAsyncCallback) {
		if (await this._mapSubscribers.has(strName)) {
			await (await this._mapSubscribers.get(strName)).push(fnAsyncCallback);
		} else {
			await this._mapSubscribers.set(strName,[fnAsyncCallback]);
		}
	}
	
	async unsubscribe(strName,fnAsyncCallback) {
		try {
			let arrSubscribers = await this._mapSubscribers.get(strName);
			let numIdx;
			while ((numIdx = arrSubscribers.indexOf(fnAsyncCallback)) >= 0) arrSubscribers.splice(numIdx,1);
		} catch(err) {}
	}
	
	async callSubscribers(strName) {
		// console.log("callSubscribers " + strName);
		let setFnCallback = await this._mapSubscribers.get(strName);
		let variable = await this._mapVars.get(strName);
		if (setFnCallback !== undefined && variable !== undefined) {
			await setFnCallback.forEach( async (fnAsyncCallback) => {
				try {
					await fnAsyncCallback(variable);
				} catch (err) {
					console.warn(`a subscriber callback for ${strName}='${valNew} failed: ${err}`);
				}
			});
		}
	}
	
	async touch(...arrStrName) {
		if (arrStrName.length <= 0) {
			await this._mapVars.forEach( async (strName,variable) => {
				this.callSubscribers(strName);
			});
		} else {
			for (const strName of arrStrName) {
				this.callSubscribers(strName);
			}
		}
	}
	
	async getFormat(strName) {
		try {
			return await (await this._mapVars.get(strName)).getFormat();
		} catch (err) {
			console.warn(`failed to get format string for '${strName}': ${err}`);
			return undefined;
		}
	}
	
	async getPrefixSuffix(strName) {
		try {
			return await (await this._mapVars.get(strName)).getPrefixSuffix();
		} catch (err) {
			console.warn(`failed to get prefix/suffix strings for '${strName}': ${err}`);
			return undefined;
		}
	}
	
	async export() {
		let mapExport = new Map();
		await this._mapVars.forEach( (strName,variable) => {
			mapExport.set(strName,variable.getValue())
		});
		return mapExport;
	}
	
	async import(mapImport) {
		for (const [strKey,value] of mapImport.entries()) {
			await this.setValue(strKey,value);
		}
	}
}
