/* Abelbeck Aviation Checklist - stack machine library script
Copyright 2024 Frank Abelbeck <frank@abelbeck.info>

This file is part of the Abelbeck Aviation Checklist (AAC) toolbox.

The AAC toolbox is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the AAC toolbox.  If not, see <http://www.gnu.org/licenses/>.
*/

import {
	AsyncIOMap,
	AsyncIOLock
} from "./libAio.js";

import {
	VariableNumber,
	VariableString
} from "./libVar.js";


export class StackMachine {
	constructor() {
		this._lockStack = new AsyncIOLock();
		this._arrStack = [];
		
		this._mapCache = new AsyncIOMap();
		
		const methods = Object.getOwnPropertyNames(Object.getPrototypeOf(this));
		methods.filter((method) => (method !== "constructor")).forEach((method) => { this[method] = this[method].bind(this);});
		
		this._mapOps = new Map([
			["+",      this.opAdd],
			["-",      this.opSub],
			["*",      this.opMul],
			["/",      this.opDiv],
			
			["//" ,    this.opIDiv],
			["/mod",   this.opIDivMod],
			["mod",    this.opMod],
			
			["pow",    this.opPow],
			
			// ["sqrt",   this.opSqareRoot],
			// ["cbrt",   this.opCubicRoot],
			// ["exp",    this.opExponential],
			
			// ["log",    this.opLogarithm],
			// ["lb",     this.opLogarithm2],
			// ["lg",     this.opLogarithm10],
			// ["ln",     this.opLogarithmE],
			
			// ["e",      this.opNumE],
			// ["pi",     this.opNumPi],
			// ["degs",   this.opDegrees],
			// ["rads",   this.opRadians],
			
			["inc",    this.opInc],
			["dec",    this.opDec],
			
			["ceil",   this.opCeiling],
			["floor",  this.opFloor],
			["trunc",  this.opTrunc],
			["round",  this.opRoundHalfToEven],
			["fract",  this.opFract],
			
			["neg",    this.opNeg],
			["sgn",    this.opSgn],
			["abs",    this.opAbs],
			["min",    this.opMin],
			["max",    this.opMax],
			
			["len",    this.opLen],
			
			// ["sin",    this.opSin],
			// ["sinh",   this.opSinh],
			// ["asin",   this.opAsin],
			// ["asinh",  this.opAsinh],
			// ["cos",    this.opCos],
			// ["acos",   this.opAcos],
			// ["cosh",   this.opCosh],
			// ["acosh",  this.opAcosh],
			// ["tan",    this.opTan],
			// ["atan",   this.opAtan],
			// ["atan2",  this.opAtan2],
			// ["tanh",   this.opTanh],
			// ["atanh",  this.opAtanh],
			
			["=",      this.opEQ],
			["!=",     this.opNE],
			[">",      this.opGT],
			[">=",     this.opGE],
			["<",      this.opLT],
			["<=",     this.opLE],
			["and",    this.opAnd],
			["or",     this.opOr],
			["xor",    this.opXor],
			["not",    this.opNot],
			["if",     this.opIf],
			["dup",    this.opDup],
			["swap",   this.opSwap],
			["rot",    this.opRot],
			["over",   this.opOver],
			["pick",   this.opPick],
			["drop",   this.opDrop],
		]);
		/*
		 * integrity check of operator dictionary
		 */
		for (const [strOp,fnOp] of this._mapOps.entries()) {
			if (fnOp === undefined) {
				console.warn(`function of operator '${strOp}' is undefined: removing it from operator dictionary`)
				this._mapOps.delete(strOp);
			}
		}
	}
	
	isIdentifier(strValue) {
		return /^[a-zA-Z_][a-zA-Z0-9_]*$/.test(strValue);
	}
	
	isOperator(strValue) {
		return this._mapOps.has(strValue);
	}
	
	isString(strValue) {
		return /^("([^"\\]|\\.)*"|'([^'\\]|\\.)*')$/.test(strValue);
	}
	
	isDecimal(strValue) {
		return /^[-+]?([0-9]|[1-9][0-9]+)(\.[0-9]+)?$/.test(strValue);
	}
	
	isHexadecimal(strValue) {
		return /^0x[0-9a-fA-F]+$/.test(strValue);
	}
	
	isOctal(strValue) {
		return /^0o[0-7]+$/.test(strValue);
	}
	
	isBinary(strValue) {
		return /^0b[01]+$/.test(strValue);
	}
	
	async clear() {
		await this._lockStack.write( async () => { this._arrStack = []; })
		await this._mapCache.clear();
	}
	
	async register(strExpression) {
		/*
		 * explicitly register an expression, i.e. compile it and return
		 * the list of dependency identifiers
		 */
		let [arrTokens,setIdentifiers] = await this.compile(strExpression);
		return setIdentifiers;
	}
	
	async compile(strExpression) {
		/*
		 * compile the given expression
		 * if the expression is recorded in _mapCache, return the entry,
		 * otherwise compile it
		 */
		let arrTokens, setIdentifiers;
		if (!(await this._mapCache.has(strExpression))) {
			/*
				* expression is not yet known: compile it
				*/
			arrTokens = [];
			setIdentifiers = new Set();
			let fnOp;
			for (const token of strExpression.split(/\s+/)) {
				if (this.isOperator(token)) {
					/*
					 * token is a known operator: create operator wrapper function and push to token list
					 *  - which pops the needed number of arguments off stack,
					 *  - executes the operator function,
					 *  - and pushes the result(s) back on stack
					 */
					arrTokens.push( this._mapOps.get(token) );
					
				} else if (this.isIdentifier(token)) {
					/*
					 * token is an identifier: record in set of identifiers and push @identifier to token list
					 */
					setIdentifiers.add(token);
					arrTokens.push("@" + token);
					
				} else if (this.isString(token)) {
					/*
					 * token is a string (enclosed by either single or double quotes, with backslash escaping)
					 * remove quotes, replace escape sequences, push as $string to token list
					 */
					arrTokens.push('$' + token
						.slice(1,-1)
						.replaceAll("\\b","\b")
						.replaceAll("\\f","\f")
						.replaceAll("\\n","\n")
						.replaceAll("\\r","\r")
						.replaceAll("\\t","\t")
						.replaceAll("\\v","\v")
						.replaceAll(/\\x([0-9a-fA-F]{2})/g, (str,p1,offset,s) => String.fromCharCode(parseInt(p1,16)) )
						.replaceAll(/\\([0-7]{1,2}(?![0-7])|[0-7]{3})/g, (str,p1,offset,s) => String.fromCharCode(parseInt(p1,8)) )
						.replaceAll(/\\u([0-9a-fA-F]{4})/g, (str,p1,offset,s) => String.fromCharCode(parseInt(p1,16)) )
						.replaceAll(/\\U([0-9a-fA-F]{8})/g, (str,p1,offset,s) => String.fromCharCode(parseInt(p1,16)) )
						.replaceAll(/\\(.)/g,"$1")
					);
					
				} else if (this.isDecimal(token)) {
					/*
					 * token is a decimal number: create VariableNumber instance and push to token list
					 */
					arrTokens.push( Number(token) );
					
				} else if (this.isHexadecimal(token)) {
					/*
					 * token is a hexadecimal number: create VariableNumber instance and push to token list
					 */
					arrTokens.push( parseInt(token,16) );
					
				} else if (this.isOctal(token)) {
					/*
					 * token is an octal number: create VariableNumber instance and push to token list
					 */
					arrTokens.push( parseInt(token,8) );
					
				} else if (this.isBinary(token)) {
					/*
					 * token is a binary number: create VariableNumber instance and push to token list
					 */
					arrTokens.push( parseInt(token,2) );
					
				} else {
					console.error(`compileExpression(): invalid token ${token}`);
					arrTokens = [];
					break;
				}
			}
			// now the token list should only contain strings (=identifiers), functions (=operators), VariableString, or VariableNumber instances.
			
			await this._mapCache.set(strExpression,[arrTokens,setIdentifiers]);
			
		} else {
			[arrTokens,setIdentifiers] = await this._mapCache.get(strExpression);
		}
		return [arrTokens,setIdentifiers];
	}
	
	async execute(strExpression, variableManager) {
		/*
		 * execute given expression;
		 * compile it if not yet doen (just in time)
		 */
		let varRet,arrProg;
		let [arrTokens,setIdentifiers] = await this.compile(strExpression);
		/*
		 * process all program tokens and modify stack
		 */
		if (arrTokens) {
			await this._lockStack.write( async () => {
				this._arrStack = []
				for (let token of arrTokens) {
					if (typeof(token) === "function") {
						try {
							token();
						} catch (err) {
							throw new Error(`op ${token} failed (${err})`);
						}
					} else if (typeof(token) === "string") {
						switch (token[0]) {
							case "@":
								/*
								* identifier string starting with @:
								* remove @, lookUp identifier in aioStore,
								* push value to stack
								*/
								let varValue = await variableManager.getValue(token.slice(1));
								if (varValue !== undefined) {
									this._arrStack.push(varValue);
								} else {
									throw new Error(`resolving identifier '${token}' failed`);
								}
								break;
							case "$":
								/*
								* regular encoded string starting with $:
								* remove $, push to stack
								*/
								token = token.slice(1);
							default:
								/*
								* neither of the above: assume irregular string;
								* push to stack unaltered
								*/
								this._arrStack.push(token)
						}
					} else {
						/*
							* anything else: just push to stack
							*/
						this._arrStack.push(token);
					}
				}
				varRet = this._arrStack[0];
			},
			async (err) => {
				// console.error(`execution of ${strExpression} failed: ${err}`);
				varRet = undefined;
			});
		}
		return varRet;
	}
	
	opPop(numItems) {
		let arrRet = [];
		while (numItems-- > 0) arrRet.unshift(this._arrStack.pop());
		return arrRet;
	}
	
	opPush(...args) {
		for (const valRes of args) {
			this._arrStack.push(valRes);
		}
	}
	
	opAdd()    {
		let [summand1,summand2] = this.opPop(2);
		switch (typeof(summand1) + typeof(summand2)) {
			case "numbernumber":
			case "stringstring":
				this.opPush(summand1 + summand2);
				break;
			default:
				throw new Error("invalid operand types");
		}
	}
	
	opSub()   {
		let [minuend,subtrahend] = this.opPop(2);
		if ((typeof(minuend) + typeof(subtrahend)) === "numbernumber") {
			this.opPush(minuend - subtrahend); 
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opMul() {
		let [factor1,factor2] = this.opPop(2);
		switch (typeof(factor1) + typeof(factor2)) {
			case "numbernumber":
				this.opPush(factor1 * factor2);
				break;
			case "numberstring":
				this.opPush(factor2.repeat(factor1));
				break;
			case "stringnumber":
				this.opPush(factor1.repeat(factor2));
				break;
			default:
				throw new Error("invalid operand types");
		}
	}
	
	opDiv() {
		let [dividend,divisor] = this.opPop(2);
		if ((typeof(dividend) + typeof(divisor)) === "numbernumber") {
			this.opPush(dividend / divisor);
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opIDiv() {
		let [dividend,divisor] = this.opPop(2);
		if ((typeof(dividend) + typeof(divisor)) === "numbernumber") {
			this.opPush(Math.floor(dividend / divisor));
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opIDivMod() { 
		let [dividend,divisor] = this.opPop(2);
		if ((typeof(dividend) + typeof(divisor)) === "numbernumber") {
			let quotient = Math.floor(dividend / divisor);
			this.opPush( quotient, dividend - divisor*quotient );
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opMod() {
		let [dividend,divisor] = this.opPop(2);
		if ((typeof(dividend) + typeof(divisor)) === "numbernumber") {
			this.opPush( dividend - divisor*Math.floor(dividend / divisor) );
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opPow() {
		let [base,power] = this.opPop(2);
		if ((typeof(base) + typeof(power)) === "numbernumber") {
			this.opPush( base ** power );
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opInc() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( operand + 1 );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opDec() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( operand - 1 );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opNeg() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( -operand );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opFloor() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.floor(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opCeiling() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.ceil(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opTrunc() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.trunc(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opRoundToZero() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.trunc(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opRoundAwayFromZero() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( (operand < 0) ? Math.floor(operand) : Math.ceil(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opRoundHalfToCeiling() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.ceil(0.5*Math.floor(2*operand)) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opRoundHalfToFloor() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.floor(0.5*Math.ceil(2*operand)) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opRoundHalfToZero() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( -Math.sign(operand) * Math.floor(-Math.abs(operand) + 0.5) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opRoundHalfAwayZero() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( -Math.sign(operand) * Math.ceil(-Math.abs(operand) - 0.5) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opRoundHalfToEven() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			/*
			 * if |fract| == 0.5: round to nearest even
			 * if |fract| > 0.5: round away from zero
			 * if |fract| < 0.5: round towards zero
			 */
			let numInt = Math.floor(operand);
			let numUp = (operand >= 0) ? operand + 1 : operand - 1;
			let numNearestEven = (numInt % 2 === 0) ? numInt : numUp;
			let numFract = Math.abs(operand - numInt);
			if (numFract == 0.5) {
				this.opPush(numNearestEven);
			} else if (numFract > 0.5) {
				this.opPush(numUp);
			} else {
				this.opPush(numInt);
			}
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opFract() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( operand - Math.trunc(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opSgn() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.sign(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opAbs() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( Math.ab(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opMin()    {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1) === typeof(operand2)) {
			this.opPush( (operand1 < operand2) ? operand1 : operand2 );
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opMax()    {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1) === typeof(operand2)) {
			this.opPush( (operand1 > operand2) ? operand1 : operand2 );
		} else {
			throw new Error("invalid operand types");
		}
	}
	
	opLen() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "string") {
			this.opPush( operand.length );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opEQ() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1) === typeof(operand2)) {
			this.opPush( Number(operand1 === operand2) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opNE() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand) === typeof(operand2)) {
			this.opPush( Number(operand1 !== operand2) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opGT() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1) === typeof(operand2)) {
			this.opPush( Number(operand1 > operand2) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opGE() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1) === typeof(operand2)) {
			this.opPush( Number(operand1 >= operand2) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opLT() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1) === typeof(operand2)) {
			this.opPush( Number(operand1 < operand2) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opLE() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1) === typeof(operand2)) {
			this.opPush( Number(operand1 <= operand2) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opAnd() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1)+typeof(operand2) === "numbernumber") {
			this.opPush( Number(Boolean(operand1) && Boolean(operand2)) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opOr() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1)+typeof(operand2) === "numbernumber") {
			this.opPush( Number(Boolean(operand1) || Boolean(operand2)) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opXor() {
		let [operand1,operand2] = this.opPop(2);
		if (typeof(operand1)+typeof(operand2) === "numbernumber") {
			let boolOperand1 = Boolean(operand1);
			let boolOperand2 = Boolean(operand2);
			this.opPush( Number(!boolOperand1 && boolOperand2 || boolOperand1 && !boolOperand2) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opNot() {
		let [operand] = this.opPop(1);
		if (typeof(operand) === "number") {
			this.opPush( !Boolean(operand) );
		} else {
			throw new Error("invalid operand type");
		}
	}
	
	opIf() {
		let [operand1,operand2,testResult] = this.opPop(3);
		if (typeof(testResult) === "number") {
			this.opPush( (testResult !== 0) ? operand1 : operand2 );
		} else {
			throw new Error("invalid test value type (not a number)");
		}
	}
	
	opDup() {
		/*
		 * duplicate the top stack item:
		 *    i0 dup ---> i0 i0
		 */
		let [item0] = this.opPop(1);
		this.opPush(item0,item0);
	}
	
	opSwap() {
		/*
		 * swap top two stack items
		 *    i1 i0 swap ---> i0 i1
		 */
		let [item1,item0] = this.opPop(2);
		this.opPush(item0,item1);
	}
	
	opRot() {
		/*
		 * pull the third stack item on top
		 *    i2 i1 i0 rot ---> i1 i0 i2
		 */
		let [item2,item1,item0] = this.opPop(3);
		this.opPush(item1,item0,item2);
	}
	
	opOver() {
		/*
		 * copy the second stack item
		 *    i1 i0 over ---> i1 i0 i1
		 */
		let [item1,item0] = this.opPop(2);
		this.opPush(item1,item0,item1);
	}
	
	opPick() {
		/*
		 * copy the stack item at a given index
		 *    iN...i0 n pick ---> iN...i0 in
		 */
		let [numIdx] = this.opPop(1);
		if (typeof(numIdx) === "number" && numIdx >= 0 && numIdx < this._arrStack.length) {
			this.opPush(this._arrStack.at(numIdx));
		} else {
			throw new Error("invalid stack index");
		}
	}
	
	opDrop() {
		/*
		 * remove the top stack item
		 *    i2 i1 i0 drop---> i2 i1
		 */
		this.opPop(1);
	}
}


