/* Abelbeck Aviation Checklist - keypad library script
Copyright 2024 Frank Abelbeck <frank@abelbeck.info>

This file is part of the Abelbeck Aviation Checklist (AAC) toolbox.

The AAC toolbox is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the AAC toolbox.  If not, see <http://www.gnu.org/licenses/>.
*/

import { WebDocumentManager } from "./libWebDoc.js";
import { AsyncIOLock } from "./libAio.js";
import { Variable } from "./libVar.js";

export class KeyPad {
	/*
	 * <article id=myKeypadId>
	 *    content goes here
	 * </article>
	 * 
	 */
	constructor(strId,reValidity) {
		this.setStyleSheet("css/keypad.css");
		this._strId = String(strId);
		this._strValue = "";
		this._strPrefix = "";
		this._strSuffix = "";
		this._numMaxChars = Infinity;
		this._boolHidePrefixIfEmpty = true;
		this._boolHideSuffixIfEmpty = true;
		
		this.setRegExpValidity(reValidity);
		
		this._lock = new AsyncIOLock();
		
		this._nodeDisplay = WebDocumentManager.constructNode("output");
		this._nodeDivButtonArea = WebDocumentManager.constructNode("div");
		this._nodeButtonEnter = WebDocumentManager.constructButton("Enter", this.wrapClickEvent(this.callbackEnter), { class:"aacKeyPadEnter" } );
		this._nodeButtonEscape = WebDocumentManager.constructButton("Esc", this.wrapClickEvent(this.callbackEscape), { class:"aacKeyPadEscape" } );
		this._nodeCurrentDiv = this._nodeDivButtonArea;
		this._nodeTitle = WebDocumentManager.constructNode("h4","");
		this._nodeMain = WebDocumentManager.constructNode("article",
			[
				this._nodeTitle,
				WebDocumentManager.constructNode("div", [
					this._nodeDisplay,
					WebDocumentManager.constructButton("&Cross;", this.wrapClickEvent(this.callbackDelete), { class:"aacKeyPadDelete" } ),
				]),
				this._nodeDivButtonArea,
				WebDocumentManager.constructNode("div", [
					this._nodeButtonEscape,
					this._nodeButtonEnter
				]),
			],
			{ id:strId, class:"aacKeyPad" }
		);
		this._callbackHookEnter = () => {};
		this._callbackHookEscape = () => {};
		
		const methods = Object.getOwnPropertyNames(Object.getPrototypeOf(this));
		methods.filter((method) => (method !== "constructor")).forEach((method) => { this[method] = this[method].bind(this);});
	}
	
	wrapClickEvent(fnCallback) {
		return async (event) => {
			try {
				navigator.vibrate(100);
			} catch (err) {}
			await this._lock.write( async () => { await fnCallback(this,event); });
		}
	}
	
	setStyleSheet(strFileStyle) {
		if (document.querySelector(`link[class='aacKeyPad'][rel='stylesheet'][href='${strFileStyle}']`) === null) {
			document.head.appendChild(WebDocumentManager.constructNode("link",null,{ class:'aacKeyPad', rel:"stylesheet", href:strFileStyle}));
		}
	}
	
	removeStyleSheet(strFileStyle) {
		let node = document.querySelector(`link[rel='stylesheet'][href='${strFileStyle}']`);
		if (node !== null) {
			node.remove();
		}
	}
	
	addMainClass(strClass) {
		this._nodeMain.classList.add(strClass);
	}
	
	setRegExpValidity(reValidity) {
		if (reValidity === undefined) {
			this._reValidity = /^.*$/;
		} else {
			this._reValidity = new RegExp(reValidity);
		}
	}
	
	setPrefix(strPrefix) {
		this._strPrefix = strPrefix;
	}
	
	setSuffix(strSuffix) {
		this._strSuffix = strSuffix;
	}
	
	setHidePrefixIfEmpty(boolValue) {
		this._boolHidePrefixIfEmpty = Booelan(boolValue);
	}
	
	setHideSuffixIfEmpty(boolValue) {
		this._boolHideSuffixIfEmpty = Booelan(boolValue);
	}
	
	async callbackEnter(self,event) {
		await self._callbackHookEnter(self.getValue());
	}
	
	async callbackEscape(self,event) {
		await self._callbackHookEscape();
	}
	
	async callbackDelete(self,event) {
		await self.setValue("");
	}
	
	isValid(strValue) {
		try {
			return this._reValidity.test(strValue);
		} catch (err) {}
		return false;
	}
	
	format(value) {
		return value;
	}
	
	parse(value) {
		return String(value);
	}
	
	processValue(strValue) {
		return strValue;
	}
	
	setValue(strValue) {
		if (typeof(strValue) !== "string") {
			strValue = this.parse(strValue);
		}
		strValue = this.processValue(strValue);
		strValue = strValue.slice(0,this._numMaxChars);
		if (this.isValid(strValue)) {
			this._nodeDisplay.classList.remove("aacKeyPadDisplayInvalid");
			this._nodeButtonEnter.removeAttribute("disabled");
		} else {
			this._nodeDisplay.classList.add("aacKeyPadDisplayInvalid");
			this._nodeButtonEnter.setAttribute("disabled","true");
		}
		
		let strPrefix = (this._boolHidePrefixIfEmpty && strValue === "") ? "" : this._strPrefix;
		let strSuffix = (this._boolHideSuffixIfEmpty && strValue === "") ? "" : this._strSuffix;
		this._nodeDisplay.innerHTML = `${strPrefix}${this.format(strValue)}${strSuffix}`;
		this._strValue = strValue;
	}
	
	getValue() {
		return this.format(this._strValue);
	}
	
	setCallbackEnter(fnCallbackEnter) {
		if (typeof(fnCallbackEnter) === "function") {
			this._callbackHookEnter = fnCallbackEnter;
		} else {
			this._callbackHookEnter = () => {};
		}
	}
	
	setCallbackEscape(fnCallbackEscape) {
		if (typeof(fnCallbackEscape) === "function") {
			this._callbackHookEscape = fnCallbackEscape;
		} else {
			this._callbackHookEscape = () => {};
		}
	}
	
	getNode() {
		return this._nodeMain;
	}
	
	addKey(strValue,fnCallback,objAttributes) {
		/*
		 * adds a button to the current div, and wires 'click' to the given
		 * callback function
		 * 
		 * the callback function is called with the current instance as first
		 * argument and the event as second argument
		 */
		let node = WebDocumentManager.constructButton(strValue, this.wrapClickEvent(fnCallback), objAttributes);
		this._nodeCurrentDiv.appendChild( node );
		return node;
	}
	
	appendNewDivision(numIdx=-1,strName) {
		let objAttr = {};
		if (strName !== undefined) {
			objAttr.id = `${this._strId}_${strName}`;
		}
		let nodeDiv = WebDocumentManager.constructNode("div",undefined,objAttr);
		
		/*
		 * CSS node indices are one-based, whereas numIdx should be zero-based
		 * thus we need to add one when selecting the nth element
		 * 
		 * numIdx     should address in order to apply insertBefore()
		 * ---------------------------------------------------------------------
		 * -2         nth-last-of-type(1)
		 * -1         nth-last-of type(0) => null => insertBefore=append
		 *  0         nth-of-type(1)
		 * +1         nth-of-type(2)
		 * +2         nth-of-type(3)
		 */
		numIdx++;
		if (numIdx <= 0) {
			this._nodeCurrentDiv.insertBefore(nodeDiv,this._nodeCurrentDiv.querySelector(`:scope > div:nth-last-of-type(${-numIdx-1})`));
		} else {
			this._nodeCurrentDiv.insertBefore(nodeDiv,this._nodeCurrentDiv.querySelector(`:scope > div:nth-of-type(${numIdx})`));
		}
		this._nodeCurrentDiv = nodeDiv;
	}
	
	getDivision(strName,numIdx) {
		if (numIdx === undefined) {
			return this._nodeDivButtonArea.querySelector(`#${this._strId}_${strName}`);
		} else {
			/*
			 * CSS node indices are one-based, whereas numIdx should be zero-based
			 * thus we need to add one when selecting the nth element
			 * 
			 * numIdx     should address in order to select the element
			 * -----------------------------------------------------------------
			 * -2         nth-last-of-type(2)
			 * -1         nth-last-of type(1)
			 *  0         nth-of-type(1)
			 * +1         nth-of-type(2)
			 * +2         nth-of-type(3)
			 */
			if (numIdx < 0) {
				/*
				 * CSS cannot understand indices of zero and below, thus we need to detect
				 * negative indices and turn them into the nth node from the back (i.e. -numIdx)
				 */
				return this._nodeDivButtonArea.querySelector(`#${this._strId}_${strName} > div:nth-last-of-type(${-numIdx})`);
			} else {
				/*
				 * CSS indices are one-based, thus we need to add 
				 */
				return this._nodeDivButtonArea.querySelector(`#${this._strId}_${strName} > div:nth-of-type(${numIdx+1})`);
			}
		}
	}
	
	setCurrentDivision(strName) {
		let nodeDiv = this.getDivision(strName);
		if (nodeDiv !== null) {
			this._nodeCurrentDiv = nodeDiv;
		}
		return nodeDiv !== null;
	}
	
	closeCurrentDivision() {
		if (this._nodeCurrentDiv !== this._nodeDivButtonArea) {
			this._nodeCurrentDiv = this._nodeCurrentDiv.parentNode;
		}
	}
	
	closeAllDivisions() {
		this._nodeCurrentDiv = this._nodeDivButtonArea;
	}
	
	showDivision(...arrStrName) {
		/*
		 * hide all direct children to nodeDivButtonArea except for the ones given as arguments
		 */
		for (const strName of arrStrName) {
			let strId = `${this._strId}_${strName}`;
			let nodeDiv = this._nodeDivButtonArea.querySelector(`#${strId}`);
			if (nodeDiv !== null) {
				for (const node of this._nodeDivButtonArea.querySelectorAll(`:scope > div:not([id="${strId}"])`)) {
					node.setAttribute("hidden","hidden");
				}
				nodeDiv.removeAttribute("hidden");
			}
		}
	}
	
	setMaxChars(numChars) {
		this._numMaxChars = numChars;
	}
	
	getMaxChars() {
		return this._numMaxChars;
	}
	
	clearKeyArea() {
		this._nodeDivButtonArea.replaceChildren();
		this._nodeCurrentDiv = this._nodeDivButtonArea;
	}
	
	setTitle(strTitle) {
		this._nodeTitle.innerHTML = (strTitle === undefined) ? "" : strTitle;
	}
	
	setup(strTitle,varValue, fnCallbackEnter, fnCallbackEscape) {
		this.setTitle(strTitle);
		this.setValue(varValue);
		this.setCallbackEnter(fnCallbackEnter);
		this.setCallbackEscape(fnCallbackEscape);
	}
}


export class KeyPadMultiLayout extends KeyPad {
	constructor(strId,reValidity) { super(strId,reValidity);
		this._strActiveLayout = "";
		this._strNewLayout = "";
		this._mapSticky = new Map(); // map sticky key code to a set of nodes
		this._mapSpecial = new Map(); // map special key identifier to a set of nodes
		this._mapStickyValue = new Map([["off","pending"],["pending","on"],["on","off"]])
		
		this._strActiveDeadKey = "";
		this._mapDeadKey = new Map();
		
		/*
		 * map special key identifiers to routines
		 * (layouts will be mapped to a switching routine later)
		 */
		this._mapSpecialRoutine = new Map();
		this.registerSpecialKeyRoutine("$newline",   (self,event) => {
			self.setValue(self._strValue + "\n");
			self.setActiveDeadKey("");
		});
		this.registerSpecialKeyRoutine("$tabulator", (self,event) => {
			self.setValue(self._strValue + "\t");
			self.setActiveDeadKey("");
		});
		this.registerSpecialKeyRoutine("$backspace", (self,event) => {
			self.setValue(self._strValue.slice(0,-1));
			self.setActiveDeadKey("");
		});
		this.registerSpecialKeyRoutine("$deadkey",   (self,event) => {
			self.setActiveDeadKey(event.target.dataset.keypadCode);
		});
	}
	
	setLayout(mapLayout,strDivisionToShow) {
		/*
		 * process layout map:
		 * 
		 * mapLayout["layouts"][""] -> normal layout
		 * mapLayout["layouts"][...] -> additional layouts
		 * mapLayout["specialKeys"] -> map key code to an identifier:
		 *    $newline: insert a newline character
		 *    $backspace: remove last character
		 *    $tabulator: insert a tabulator character
		 *    nameOfALayout: switch to given layout
		 * mapLayout["stickyKeys"] -> set of key codes that can be checked when click two times in a row
		 */
		try {
			if (!mapLayout.has("layouts")) mapLayout.set("layouts",new Map());
			if (!mapLayout.has("specialKeys")) mapLayout.set("specialKeys",new Map());
			if (!mapLayout.has("stickyKeys")) mapLayout.set("stickyKeys",new Set());
			if (!mapLayout.has("deadKeys")) mapLayout.set("deadKeys",new Set());
		} catch (err) {
			return;
		}
		
		for (const [strDeadKey,mapDeadKey] of mapLayout.get("deadKeys").entries()) {
			for (const [strKeyIn,strKeyOut] of mapDeadKey.entries()) {
				this._mapDeadKey.set(strDeadKey+strKeyIn,strKeyOut);
				mapLayout.get("specialKeys").set(strDeadKey,"$deadkey");
			}
		}
		
		this.clearKeyArea();
		
		for (const [strLayoutName,arrLayout] of mapLayout.get("layouts").entries()) {
			/*
			 * create a new division and iterate over rows and columns of the layout
			 */
			this.appendNewDivision(-1,strLayoutName);
			
			// register special key identifier for layout switching
			this.registerSpecialKeyRoutine(strLayoutName, (self,event) => {
				if (event.target.dataset.keypadSticky !== "off") {
					self._strNewLayout = strLayoutName;
				}
			});
			
			for (const arrLine of arrLayout) {
				this.appendRow(-1,arrLine,mapLayout.get("specialKeys"),mapLayout.get("stickyKeys"));
			}
			
			this.closeCurrentDivision();
		}
		
		this.showDivision( (strDivisionToShow === undefined) ? "" : strDivisionToShow);
	}
	
	appendRow(numRow,arrLine,mapSpecialKeys,setStickyKeys) {
		this.appendNewDivision(numRow);
		
		let strSpecial, boolSticky, objAttr, node;
		for (const strKey of arrLine) {
			/*
			 * check if key is a special key and pick the matching callback
			 */
			try {
				strSpecial = mapSpecialKeys.get(strKey);
			} catch (err) {
				strSpecial = undefined;
			}
			try {
				boolSticky = setStickyKeys.has(strKey);
			} catch (err) {
				boolSticky = false;
			}
			
			objAttr = { "data-keypad-code": strKey };
			if (strSpecial !== undefined) {
				objAttr["data-keypad-special"] = strSpecial;
			}
			if (boolSticky) {
				objAttr["data-keypad-sticky"] = "off";
			}
			
			node = this.addKey(strKey, this.callbackKey, objAttr);
			
			if (node !== null) {
				if (strSpecial !== undefined) {
					this.addSpecial(strSpecial,node);
				}
				if (boolSticky) {
					this.addSticky(strKey,node);
				}
			}
		}
		
		this.closeCurrentDivision();
	}
	
	removeRow(strLayout,numRow) {
		let node = this.getDivision(strLayout,numRow);
		if (node !== null) {
			node.remove()
		}
	}
	
	addSpecial(strSpecial,node) {
		if (!this._mapSpecial.has(strSpecial)) {
			this._mapSpecial.set(strSpecial,new Set());
		}
		this._mapSpecial.get(strSpecial).add(node);
	}
	
	addSticky(strKey,node) {
		if (!this._mapSticky.has(strKey)) {
			this._mapSticky.set(strKey,new Set());
		}
		this._mapSticky.get(strKey).add(node);
	}
	
	resetNode(node) {
		for (const setSpecial of this._mapSpecial.values()) {
			setSpecial.delete(node);
		}
		for (const setSticky of this._mapSticky.values()) {
			setSticky.delete(node);
		}
		node.removeAttribute("data-keypad-sticky");
		node.removeAttribute("data-keypad-special");
	}
	
	getKey(strLayout,numRow,numColumn) {
		let nodeRow = this.getDivision(strLayout,numRow);
		let nodeCell = null;
		if (numColumn !== undefined && nodeRow !== null) {
			if (numColumn === 0) {
				nodeCell = nodeRow.firstChild;
			} else if (numColumn < 0) {
				nodeCell = nodeRow.querySelector(`:scope > button:nth-last-of-type(${-numColumn})`);
			} else {
				nodeCell = nodeRow.querySelector(`:scope > button:nth-of-type(${numColumn+1})`);
			}
		}
		return nodeCell;
	}
	
	changeKey(strLayout,numRow,numColumn,strKey,strSpecial="",boolSticky=false,fnRoutineSpecial) {
		/*
		 * get node at [layout][row][colum]
		 * 
		 * delete node from this._mapSpecial
		 * delete node from this._mapSticky
		 * 
		 * set content and data-code to strKey
		 * if strSpecial !== "": record node in this._mapSpecial[strSpecial]
		 * if boolSticky: record node in this._mapSticky;
		 * if fnRoutineSpecial !== undefined and strSpecial != "": call registerSpecialKeyRoutine for strSpecial
		 */
		
		let node = this.getKey(strLayout,numRow,numColumn);
		if (node !== null) {
			this.resetNode(node);
			node.innerHTML = strKey;
			node.dataset.keypadCode = strKey;
			if (strSpecial && typeof(strSpecial) === "string" && strSpecial > "") {
				node.dataset.keypadSpecial = strSpecial;
				this.addSpecial(strSpecial);
				if (fnRoutineSpecial && typeof(fnRoutineSpecial) === "function") {
					this.registerSpecialKeyRoutine(strSpecial,fnRoutineSpecial);
				}
			}
			if (boolSticky) {
				node.dataset.keypadSticky = "off";
				this.addSticky(strKey);
			}
		}
	}
	
	removeKey(strLayout,numRow,numColumn) {
		let node = this.getKey(strLayout,numRow,numColumn);
		if (node !== null) {
			node.remove();
		}
	}
	
	registerSpecialKeyRoutine(strName,fnRoutine) {
		this._mapSpecialRoutine.set(strName,fnRoutine);
	}
	
	async callbackKey(self,event) {
		/*
		 * general/central key press handling callback
		 */
		let strKey = event.target.dataset.keypadCode;
		let strSpecial = event.target.dataset.keypadSpecial;
		let strSticky = event.target.dataset.keypadSticky;
		self._strNewLayout = "";
		
		if (strSticky !== undefined) {
			/*
			 * sticky key: rotate state and syncronise all similar keys
			 */
			self.setSticky(event.currentTarget);
		} else if (strSpecial !== undefined && strSpecial[0] != "$") {
			self.toggleChecked(event.currentTarget);
		}
		
		if (self._mapSpecialRoutine.has(strSpecial)) {
			self._mapSpecialRoutine.get(strSpecial)(self,event);
		} else {
			/*
			 * enter a literal character
			 * if a dead key is active, modify the character
			 */
			if (self._strActiveDeadKey.length > 0) {
				strKey = self.applyDeadKey(strKey);
			}
			self.setValue(self._strValue + strKey);
		}
		
		let nodeLayout;
		let boolChecked = false;
		strSticky = "off";
		if (self._mapSpecial.has(self._strActiveLayout)) {
			nodeLayout = self._mapSpecial.get(self._strActiveLayout).values().next().value;
			strSticky = nodeLayout.dataset.keypadSticky;
			boolChecked = nodeLayout.hasAttribute("checked");
		}
		
		if (self._strNewLayout !== self._strActiveLayout) {
			/*
			 * layout change requested: only do if...
			 * 
			 * ...current layout is not associated with a sticky button
			 * ...current layout is associated with a sticky button, but...
			 *    ... new layout would be "" and sticky button is not "on", or
			 *    ... new layout is not "" and sticky button is not "off"
			 */
			if (nodeLayout === undefined) {
				self._strActiveLayout = self._strNewLayout;
				self.showDivision(self._strActiveLayout);
			} else if (self._strNewLayout === "" && strSticky !== "on" || self._strNewLayout !== "" && strSticky !== "off") {
				self._strActiveLayout = self._strNewLayout;
				self.setSticky(nodeLayout,"off");
				self.toggleChecked(nodeLayout,false);
				self.showDivision(self._strActiveLayout);
			}
		} else if (nodeLayout && (strSticky === "off" || strSticky == undefined && !boolChecked)) {
			/*
			 * no layout change requested:
			 * 
			 *    if there is a sticky button associated with this layout, and its current state is "off", fall back to "" layout
			 *
			 *    if there is a special button associated with this layout, and it is checked, fall back to "" layout
			 */
			self._strActiveLayout = "";
			self.showDivision(self._strActiveLayout);
		}
	}
	
	setSticky(nodeKey,strNewValue) {
		if (nodeKey) {
			if (strNewValue === undefined) {
				strNewValue = this._mapStickyValue.get(nodeKey.dataset.keypadSticky);
			}
			if (nodeKey.dataset.keypadSticky !== undefined && strNewValue !== undefined) {
				for (const node of this._mapSticky.get(nodeKey.dataset.keypadCode)) {
					node.dataset.keypadSticky = strNewValue;
				}
			}
		}
	}
	
	toggleChecked(nodeKey,boolState) {
		if (nodeKey) {
			if (boolState === undefined) {
				for (const node of this._mapSpecial.get(nodeKey.dataset.keypadSpecial)) {
					node.toggleAttribute("checked");
				}
			} else if (boolState) {
				for (const node of this._mapSpecial.get(nodeKey.dataset.keypadSpecial)) {
					node.setAttribute("checked","true");
				}
			} else {
				for (const node of this._mapSpecial.get(nodeKey.dataset.keypadSpecial)) {
					node.removeAttribute("checked");
				}
			}
		}
	}
	
	setActiveDeadKey(strKey) {
		let strDeadKeyOld = this._strActiveDeadKey;
		if (strKey !== "" && this._strActiveDeadKey === strKey) {
			// dead key was already active: produce a literal strKey
			this.setValue(this._strValue + strKey);
			this._strActiveDeadKey = "";
		} else {
			// set new active dead key (overriding any previous active key)
			this._strActiveDeadKey = strKey;
		}
		
		if (this._strActiveDeadKey === "" && strDeadKeyOld !== "") {
			for (const node of this._nodeDivButtonArea.querySelectorAll(`button[data-keypad-code="${strDeadKeyOld}"]`)) {
				node.removeAttribute("checked");
			}
		} else {
			for (const node of this._nodeDivButtonArea.querySelectorAll(`button[data-keypad-code="${strKey}"]`)) {
				node.setAttribute("checked","true");
			}
		}
	}
	
	applyDeadKey(strKey) {
		let strKeyModified = this._mapDeadKey.get(this._strActiveDeadKey+strKey);
		if (strKeyModified !== undefined) {
			strKey = strKeyModified;
		}
		this.setActiveDeadKey("");
		return strKey;
	}
}


export class KeyPadString extends KeyPadMultiLayout {
	
	static KeyboardLayout = new Map([
		["en", new Map([
			["layouts", new Map([
				["", [
					["`","1","2","3","4","5","6","7","8","9","0","-","=","&DoubleLeftArrow;"],
					["&RightArrowBar;","q","w","e","r","t","y","u","i","o","p","[","]","\\"],
					["a","s","d","f","g","h","j","k","l",";","'","&ldsh;"],
					["&DoubleUpArrow;","z","x","c","v","b","n","m",",",".","/"],
					[" "]
				]],
				["shift", [
					["~","!",'@',"#","$","%","^","&","*","(",")","_","+","&DoubleLeftArrow;"],
					["&RightArrowBar;","Q","W","E","R","T","Y","U","I","O","P","{","}","|"],
					["A","S","D","F","G","H","J","K","L",":",'"',"&ldsh;"],
					["&DoubleUpArrow;","Z","X","C","V","B","N","M","<",">","?"],
					[" "]
				]],
			])],
			["specialKeys", new Map([
				["&DoubleUpArrow;","shift"],
				["&DoubleLeftArrow;","$backspace"],
				["&ldsh;","$newline"],
				["&RightArrowBar;","$tabulator"]
			])],
			["stickyKeys", new Set(["&DoubleUpArrow;"])]
		])],
		["de", new Map([
			["layouts", new Map([
				["", [
					["^","1","2","3","4","5","6","7","8","9","0","ß","´","&DoubleLeftArrow;"],
					["&RightArrowBar;","q","w","e","r","t","z","u","i","o","p","ü","+"],
					["a","s","d","f","g","h","j","k","l","ö","ä","#","&ldsh;"],
					["<","y","x","c","v","b","n","m",",",".","-"],
					["&DoubleUpArrow;"," ","AltGr"]
				]],
				["shift", [
					["°","!",'"',"§","$","%","&","/","(",")","=","?","`","&DoubleLeftArrow;"],
					["&RightArrowBar;","Q","W","E","R","T","Z","U","I","O","P","Ü","*"],
					["A","S","D","F","G","H","J","K","L","Ö","Ä","'","&ldsh;"],
					[">","Y","X","C","V","B","N","M",";",":","_"],
					["&DoubleUpArrow;"," ","AltGr"]
				]],
				["altgr", [
					["","¹","²","³","","","","{","[","]","}","\\","","&DoubleLeftArrow;"],
					["&RightArrowBar;","@","","€","","","","","","","","","~"],
					["","","","","","","","","","","","","&ldsh;"],
					["|","","","","","","","","","",""],
					["&DoubleUpArrow;"," ","AltGr"]
				]],
			])],
			["specialKeys", new Map([
				["&DoubleUpArrow;","shift"],
				["AltGr","altgr"],
				["&DoubleLeftArrow;","$backspace"],
				["&ldsh;","$newline"],
				["&RightArrowBar;","$tabulator"]
			])],
			["stickyKeys", new Set(["&DoubleUpArrow;" /*,"AltGr" */])],
			["deadKeys", new Map([
				["´", new Map([
					["a","á"],
					["A","Á"],
					["c","ć"],
					["C","Ć"],
					["e","é"],
					["E","É"],
					["i","í"],
					["I","Í"],
					["l","ĺ"],
					["L","Ĺ"],
					["n","ń"],
					["N","Ń"],
					["o","ó"],
					["O","Ó"],
					["u","ú"],
					["U","Ú"],
					["y","ý"],
					["Y","Ý"],
					["z","ź"],
					["Z","Ź"],
				])],
				["`", new Map([
					["a","à"],
					["A","À"],
					["e","è"],
					["E","È"],
					["i","ì"],
					["I","Ì"],
					["o","ò"],
					["O","Ò"],
					["u","ù"],
					["U","Ù"],
					["w","ẁ"],
					["W","Ẁ"],
					["y","ý"],
					["Y","Ý"],
				])],
				["^", new Map([
					["a","â"],
					["A","Â"],
					["c","ĉ"],
					["C","Ĉ"],
					["e","ê"],
					["E","Ê"],
					["g","ĝ"],
					["G","Ĝ"],
					["h","ĥ"],
					["H","Ĥ"],
					["i","î"],
					["I","Î"],
					["j","ĵ"],
					["J","Ĵ"],
					["o","ô"],
					["O","Ô"],
					["s","ŝ"],
					["S","Ŝ"],
					["u","û"],
					["U","Û"],
					["w","ŵ"],
					["W","Ŵ"],
					["y","ŷ"],
					["Y","Ŷ"],
				])],
				["~", new Map([
					["a","ã"],
					["A","Ã"],
					["i","ĩ"],
					["I","Ĩ"],
					["n","ñ"],
					["N","Ñ"],
					["o","õ"],
					["O","Õ"],
				])],
			])],
		])],
		["fr", new Map([
			["layouts", new Map([
				["", [
					["²","&","é",'"',"'","(","-","è","&ndash;","ç","à",")","=","&DoubleLeftArrow;"],
					["&RightArrowBar;","a","z","e","r","t","y","u","i","o","p","^","$"],
					["q","s","d","f","g","h","j","k","l","m","ù","*","&ldsh;"],
					["<","w","x","c","v","b","n",",",";",":","!"],
					["&DoubleUpArrow;"," ","AltGr"]
				]],
				["shift", [
					["","1","2","3","4","5","6","7","8","9","0","°","+","&DoubleLeftArrow;"],
					["&RightArrowBar;","A","Z","E","R","T","Y","U","I","O","P","¨","£"],
					["Q","S","D","F","G","H","J","K","L","M","%","µ","&ldsh;"],
					[">","W","X","C","V","B","N","?",".","/","§"],
					["&DoubleUpArrow;"," ","AltGr"]
				]],
				["altgr", [
					["","","~","#","{","[","|","`","\\","&Hat;","@","]","}","&DoubleLeftArrow;"],
					["&RightArrowBar;","","","€","","","","","","","","","¤"],
					["","","","","","","","","","","","","&ldsh;"],
					["","","","","","","","","","",""],
					["&DoubleUpArrow;"," ","AltGr"]
				]],
			])],
			["specialKeys", new Map([
				["&DoubleUpArrow;","shift"],
				["AltGr","altgr"],
				["&DoubleLeftArrow;","$backspace"],
				["&ldsh;","$newline"],
				["&RightArrowBar;","$tabulator"]
			])],
			["stickyKeys", new Set(["&DoubleUpArrow;" /*,"AltGr" */])],
			["deadKeys", new Map([
				["^", new Map([
					["a","â"],
					["A","Â"],
					["e","ê"],
					["E","Ê"],
					["i","î"],
					["I","Î"],
					["o","ô"],
					["O","Ô"],
					["u","û"],
					["U","Û"],
					["y","ŷ"],
					["Y","Ŷ"],
				])],
				["¨", new Map([
					["a","̈a"],
					["A","̈A"],
					["e","̈e"],
					["E","̈E"],
					["i","̈i"],
					["I","̈I"],
					["o","ö"],
					["O","̈̈O"],
					["u","̈u"],
					["U","̈U"],
					["y","̈y"],
					["Y","̈Y"],
				])],
			])],
		])]
	]);
	
	constructor(strId,strLocale,reValidity) { super(strId,reValidity);
		this.addMainClass("aacKeyPadString");
		this.setLocale(strLocale);
	}
	
	setLocale(strLocale) {
		if (strLocale === undefined) {
			try {
				this._strLocale = navigator.language;
			} catch (err) {
				this._strLocale = "en";
			}
		}
		let mapLayout = KeyPadString.KeyboardLayout.get(this._strLocale);
		if (mapLayout === undefined) {
			/*
			 * fallback: US keyboard layout, i.e. ascii
			 */ 
			this._strLocale = "en";
			mapLayout = KeyPadString.KeyboardLayout.get("en");
		}
		this.setLayout( mapLayout,"" );
	}
}


export class KeyPadNumberBasic extends KeyPadMultiLayout {
	constructor(strId,strSeparator,reValidity) { super(strId,reValidity);
		this.addMainClass("aacKeyPadNumber");
		this._boolLeadingZeros = false;
		this._boolAutomaticSeparator = false;
		
		let mapLayout = new Map([
			["layouts", new Map([
				["",[
					["1","2","3"],
					["4","5","6"],
					["7","8","9"],
				]]
			])],
			["specialKeys", new Map([
				["&DoubleLeftArrow;","$backspace"],
			])],
		]);
		
		if (strSeparator === undefined) {
			mapLayout.get("layouts").get("").push(
				["0","&DoubleLeftArrow;"]
			);
			this._strSeparator = undefined;
			this._numMaxCharsAfterSeparator = 0;
		} else {
			mapLayout.get("layouts").get("").push(
				[strSeparator,"0","&DoubleLeftArrow;"]
			);
			mapLayout.get("specialKeys").set(strSeparator,"$separator");
			this.registerSpecialKeyRoutine("$separator", async (self,event) => { 
				/*
				 * deal with separators
				 * case: the user just entered a separator
				 */
				let strValue = self._strValue;
				if (strValue.indexOf(self._strSeparator) < 0) {
					// no separator found: append
					strValue = strValue + self._strSeparator;
				}
				if (strValue === self._strSeparator) {
					// user entered only a separator: prepend zero
					strValue = "0" + strValue;
				}
				self.setValue(strValue);
			});
			this._strSeparator = strSeparator;
			this._numMaxCharsAfterSeparator = Infinity;
		} 
		this.setLayout(mapLayout);
	}
	
	setMaxCharsAfterSeparator(numMaxCharsAfterSeparator) {
		this._numMaxCharsAfterSeparator = numMaxCharsAfterSeparator;
	}
	
	setLeadingZeros(boolLeadingZeros) {
		this._boolLeadingZeros = Boolean(boolLeadingZeros);
	}
	
	setAutomaticSeparator(boolAutomaticSeparator) {
		this._boolAutomaticSeparator = Boolean(boolAutomaticSeparator);
	}
	
	processValue(strValue) {
		if (!this._boolLeadingZeros) {
			// no leading zeros allowed: replace them with a single zero, remove single zero if followed by 1..9
			strValue = strValue.replace(/^0+$/,"0").replace(/^0+(?=[1-9])/,"");
		}
		if (this._strSeparator !== undefined) {
			let numIdxSeparator = strValue.indexOf(this._strSeparator);
			if (numIdxSeparator >= 0) {
				/*
				 * user entered a separator:  check if maximum number of chars
				 * after the separator is reached: in that case, trim value string
				 * 
				 *      .------------- numIdxSeparator
				 *      |   .--------- numIdxSeparator + _strSeparator.length + _numMaxCharsAfterSeparator
				 *      v   v
				 * xxxxx.yyy|yyy
				 * 
				 * xxxxx.yy ---> idx 5, len 1, aft 2 => max = 5 + 1 + 2 = 8
				 */
				strValue = strValue.slice(0,numIdxSeparator + this._strSeparator.length + this._numMaxCharsAfterSeparator);
				
			} else if (this._boolAutomaticSeparator) {
				/*
				 * user hasn't yet entered a separator, but the keypad should
				 * add one automatically
				 * 
				 *     .------------- numIdxSeparator = _numMaxChars - _numMaxCharsAfterSeparator - _strSeparator.length
				 *     |   .--------- _numMaxChars
				 *     v   |
				 * xxxx.yyy|
				 * 
				 * let parent method implementation deal with strings exceeding numMaxChars
				 * 
				 * xxxxx.yy ---> max 8, aft 2; len 1 => idx = 8 - 2 - 1 = 5
				 * 0    5
				 */
				numIdxSeparator = this._numMaxChars - this._numMaxCharsAfterSeparator - this._strSeparator.length;
				if (strValue.length > numIdxSeparator) {
					strValue = strValue.slice(0,numIdxSeparator) + this._strSeparator + strValue.slice(numIdxSeparator);
				}
			}
		}
		return strValue;
	}
	
	parse(value) {
		return (isNaN(value)) ? "" : String(value);
	}
}


export class KeyPadNumber extends KeyPadNumberBasic {
	constructor(strId,reValidity) {
		super(
			strId,
			".",
			(reValidity !== undefined) ? reValidity : /^$|^[+-]?([0-9]+)(.[0-9]+)?$/
		);
	}
}

export class KeyPadInteger extends KeyPadNumberBasic {
	constructor(strId,reValidity) {
		super(
			strId,
			undefined,
			(reValidity !== undefined) ? reValidity : /^$|^[+-]?([0-9]+)?$/
		);
		this.addMainClass("aacKeyPadInteger");
	}
}

export class KeyPadOperatingHours extends KeyPadNumberBasic {
	constructor(strId) {
		super(
			strId,
			":",
			/^$|^([+-]?[0-9]+)(:[0-5][0-9])?$/
		);
		this.addMainClass("aacKeyPadOpHrs");
		this.setLeadingZeros(false);
		this.setSuffix(" h");
		this.setMaxChars(8); // hhhhh:mm
		this.setMaxCharsAfterSeparator(2); // hhhhh:mm
		this.setAutomaticSeparator(true);
	}
	
	calculateValue(strValue) {
		let [strHours,strMinutes] = strValue.split(this._strSeparator);
		let intHours = parseInt(strHours);
		let intMinutes = parseInt(strMinutes);
		if (isNaN(strMinutes)) {
			intMinutes = 0;
		}
		return intHours*60 + intMinutes;
	}
	
	parse(value) {
		switch (typeof(value)) {
			case "number":
				if (isNaN(value)) {
					value = "";
				} else {
					let [intHours,intMinutes] = Variable.divmod(value,60);
					if (intMinutes < 10) {
						value = `${intHours}:0${intMinutes}`
					} else {
						value = `${intHours}:${intMinutes}`
					}
				}
				break;
			case "string":
				console.log(value.match(this._reValidity)); 
				break;
			default:
				value = "";
		}
		return value;
	}
	
	async callbackEnter(self,event) {
		/*
		 * user might have entered just a number, i.e. only hours
		 * detect and convert to hours:minutes
		 */
		let value = self.getValue();
		if (value.indexOf(self._strSeparator) < 0) {
			value = value + ":00";
		}
		await self._callbackHookEnter(value);
	}
}

export class KeyPadTime extends KeyPadOperatingHours {
	constructor(strId) { super(strId);
		this.setLeadingZeros(true);
		this.setMaxChars(5); // hh:mm
		this.setMaxCharsAfterSeparator(2); // hh:mm
		this.setAutomaticSeparator(true);
		this.setSuffix("");
		
		this.setCurrentDivision("");
		this.appendRow(0,["DEC","NOW","INC"],new Map([["DEC","$timeDecrement"],["NOW","$timeNow"],["INC","$timeIncrement"]]));
		this.registerSpecialKeyRoutine("$timeDecrement", async (self,event) => { 
			// increment value: 
			let numValue = this.calculateValue(this._strValue);
			this.setValue(numValue - 1);
		});
		this.registerSpecialKeyRoutine("$timeIncrement", async (self,event) => { 
			// decrement value
			let numValue = this.calculateValue(this._strValue);
			this.setValue(numValue + 1);
		});
		this.registerSpecialKeyRoutine("$timeNow", async (self,event) => { 
			// get current time
			let dateNow = new Date();
			this.setValue(dateNow.getUTCHours()*60 + dateNow.getUTCMinutes());
		});
	}
	
	parse(value) {
		value = Variable.modulo(value,1440);
		return super.parse(value);
	}
}


export class KeyPadQnh extends KeyPadNumber {
	constructor(strId) { super(strId)
		this.setMaxChars(4);
		this.setLeadingZeros(false);
		this.changeKey("",3,0,"ISA","$isa",false, (self,event) => { self.setValue(1013); });
	}
}


export class KeyPadRunway extends KeyPadInteger {
	constructor(strId) { super(strId,/^$|^(3[0-6]|[12][0-9]|0[1-9])$/);
		this.setMaxChars(2);
		this.setLeadingZeros(true);
	}
}


export class KeyPadSquawk extends KeyPadNumber {
	constructor(strId) { super(strId);
		this.addMainClass("aacKeyPadSquawk");
		this.setMaxChars(4);
		this.setLeadingZeros(true);
		this.changeKey("",2,1,"0"); // no need for an 8, replace with 0
		this.changeKey("",2,2,"&DoubleLeftArrow;","$backspace");
		this.changeKey("",3,0,"VFR","$vfr",false, (self,event) => { self.setValue(7000); });
		this.removeKey("",3,1); // remove surplus keys
		this.removeKey("",3,1); // remove surplus keys
	}
}


export class KeyPadHeading extends KeyPadInteger {
	constructor(strId) { super(strId,/^$|^(360|3[0-5][0-9]|[0-2][0-9]{2})$/);
		this.setMaxChars(3);
		this.setLeadingZeros(true);
	}
}


export class KeyPadAltitude extends KeyPadNumber {
	constructor(strId) { super(strId);
		this.setMaxChars(5);
		this.setLeadingZeros(false);
		
		this.changeKey("",3,0,"FL","$alt",false, (self,event) => {
			if (event.target.innerHTML === "FL") {
				event.target.innerHTML = "ALT";
				self.setMaxChars(3);
				self.setLeadingZeros(true);
				self.setPrefix("FL&nbsp;");
			} else {
				event.target.innerHTML = "FL";
				self.setMaxChars(5);
				self.setLeadingZeros(false);
				self.setPrefix("");
			}
			self.setValue("");
		});
	}
	
	processValue(strValue) {
		let node = this.getKey("",3,0);
		let strMode = node.innerHTML;
		if (strValue.startsWith("FL")) {
			if (strMode === "FL") {
				// value is a flight level, mode is ALT: switch layout
				node.innerHTML = "ALT";
				this.setMaxChars(3);
				this.setLeadingZeros(true);
				this.setPrefix("FL&nbsp;");
			}
			strValue = strValue.replace(/FL(&nbsp;|\s)+/,"");
		}
		return strValue;
	}
	
	async callbackEnter(self,event) {
		let value = self.getValue();
		if (self._strPrefix === "FL&nbsp;") {
			if (value.length > 0) {
				value = `FL ${value}`;
			} else {
				value = "";
			}
		}
		await self._callbackHookEnter(value);
	}
}


export class KeyPadAtoZ extends KeyPadMultiLayout {
	static KeyboardLayout = new Map([
		["de", new Map([
			["layouts", new Map([
				["", [
					["Q","W","E","R","T","Z","U","I","O","P"],
					["A","S","D","F","G","H","J","K","L"],
					["Y","X","C","V","B","N","M","&DoubleLeftArrow;"],
				]],
			])],
			["specialKeys", new Map([
				["&DoubleLeftArrow;","$backspace"],
			])],
		])],
		["en", new Map([
			["layouts", new Map([
				["", [
					["Q","W","E","R","T","Y","U","I","O","P"],
					["A","S","D","F","G","H","J","K","L"],
					["Z","X","C","V","B","N","M","&DoubleLeftArrow;"],
				]],
			])],
			["specialKeys", new Map([
				["&DoubleLeftArrow;","$backspace"],
			])],
		])],
		["fr", new Map([
			["layouts", new Map([
				["", [
					["A","Z","E","R","T","Y","U","I","O","P"],
					["Q","S","D","F","G","H","J","K","L","M"],
					["W","X","C","V","B","N","&DoubleLeftArrow;"],
				]],
			])],
			["specialKeys", new Map([
				["&DoubleLeftArrow;","$backspace"],
			])],
		])]
	]);
	
	constructor(strId,strLocale) { super(strId,/^$|^[A-Z]$/);
		this.addMainClass("aacKeyPadAtis");
		this.setLocale(strLocale);
		this.setMaxChars(1);
	}
	
	setLocale(strLocale) {
		if (strLocale === undefined) {
			try {
				this._strLocale = navigator.language.slice(0,2);
			} catch (err) {
				this._strLocale = "en";
			}
		}
		let mapLayout = KeyPadAtoZ.KeyboardLayout.get(this._strLocale);
		if (mapLayout === undefined) {
			/*
			 * fallback: US keyboard layout, i.e. ascii
			 */ 
			this._strLocale = "en";
			mapLayout = KeyPadAtoZ.KeyboardLayout.get("en");
		}
		this.setLayout( mapLayout,"" );
	}
}


export class KeyPadManager {
	
	static mapTypes = new Map([
		["string",         KeyPadString],
		["number",         KeyPadNumber],
		["integer",        KeyPadInteger],
		["opHrs",          KeyPadOperatingHours],
		["time",           KeyPadTime],
		["qnh",            KeyPadQnh],
		["runway",         KeyPadRunway],
		["squawk",         KeyPadSquawk],
		["heading",        KeyPadHeading],
		["altitude",       KeyPadAltitude],
		["AtoZ",           KeyPadAtoZ],
	]);
	
	constructor() {
		this._lock = new AsyncIOLock();
		this._mapKeypads = new Map();
		this._mapVars = new Map();
		
		const methods = Object.getOwnPropertyNames(Object.getPrototypeOf(this));
		methods.filter((method) => (method !== "constructor")).forEach((method) => { this[method] = this[method].bind(this);});
	}
	
	async add(strName,strKeyPad) {
		await this._lock.write( async () => {
			if (!this._mapKeypads.has(strKeyPad)) {
				/*
				 * register new type/keypad combo
				 */
				let clsKeyPad = KeyPadManager.mapTypes.get(strKeyPad);
				if (clsKeyPad !== undefined) {
					this._mapKeypads.set(strKeyPad,new clsKeyPad(`aacKeyPad_${strKeyPad}`));
				} else {
					console.warn(`unknown keypad '${strKeyPad}'`);
				}
			}
			/*
			 * map name to type
			 */
			this._mapVars.set(strName,strKeyPad);
		});
	}
	
	async clear() {
		await this._lock.write( async () => {
			this._mapKeypads.clear();
			this._mapVars.clear();
		})
	}
	
	async get(strName) {
		let keypad;
		await this._lock.read( async () => {
			keypad = this._mapKeypads.get(this._mapVars.get(strName));
		})
		return keypad;
	}
	
	async installNodes(nodeMain,objAttributes) {
		await this._lock.read( async () => {
			for (const keypad of this._mapKeypads.values()) {
				let node = keypad.getNode();
				WebDocumentManager.addAttributes(node,objAttributes);
				if (nodeMain.querySelector(`#${node.id}`) === null) {
					nodeMain.appendChild(node);
				}
			}
		})
	}
}
