/* Abelbeck Aviation Checklist - checklist module script
Copyright 2024 Frank Abelbeck <frank@abelbeck.info>

This file is part of the Abelbeck Aviation Checklist (AAC) toolbox.

The AAC toolbox is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the AAC toolbox.  If not, see <http://www.gnu.org/licenses/>.
*/

import {
	AsyncIOLock,
	AsyncIOMap,
} from "./libAio.js";

import {
	VariableManager
} from "./libVar.js";

import {
	KeyPadManager
} from "./libKeyPad.js";

import {
	AacWebDocumentManager
} from "./libAAC.js";


/*
 * define header (ATC buttons)
 */
let strHtmlHeader = `
<button data-input=atcQNH      data-prefix=QNH      data-keypad=qnh      type=button></button>
<button data-input=atcRunway   data-prefix=Runway   data-keypad=runway   type=button></button>
<button data-input=atcSquawk   data-prefix=Squawk   data-keypad=squawk   type=button></button>
<button data-input=atcHeading  data-prefix=Heading  data-keypad=heading  type=button></button>
<button data-input=atcAltitude data-prefix=Altitude data-keypad=altitude type=button></button>
<button data-input=atcATIS     data-prefix=ATIS     data-keypad=AtoZ     type=button></button>
`;

/*
 * define footer (menu/nav buttons)
 */
let strHtmlFooter = `
<button id=btnPrev      type=button>&blacktriangleleft;</button>
<button id=btnToDo      type=button>!&Congruent;</button>
<button id=btnCurrent   type=button>&blacktriangleright;&Congruent;&blacktriangleleft;</button>
<button id=btnMenu      type=button>&vellip;</button>
<button id=btnEmergency type=button></button>
<button id=btnNext      type=button>&blacktriangleright;</button>
`;

/*
 * define file menu
 */
let strHtmlMenu = `
<h4>Menu</h4>
<div>
	<button id=btnSave  type=button>Save...</button>
	<button id=btnLoad  type=button>Load...</button>
	<button id=btnClear type=button>Clear</output>
</div>
<div>
	<button id=btnInfo  type=button>About/Help...</button>
	<button id=btnTheme type=button>Theme</button>
</div>
<div>
	<button id=btnQuit type=button>Quit</button>
</div>
`;


/*
 * define normal procedure menu
 */
let strHtmlNormalList = `
<h4 data-list=normal>Normal Procedures</h4>
<ul id=listNormal>
</ul>
`;

/*
 * define emergency procedure menu
 */
let strHtmlEmergencyList = `
<h4 data-list=emergency>EMERGENCY PROCEDURES</h4>
<ul id=listEmergency>
</ul>
`;

/*
 * define info screen
 */
let strHtmlInfo = `
<style>
samp {
	display: inline-block;
	width: 2em;
	font-weight: bold;
	font-family: monospace;
	text-align: center;
	white-space: pre;
	background: var(--backgroundItem);
	border: var(--borderWidth) solid var(--text);
}
.ok   { background: var(--backgroundOk); }
.warn { background: var(--backgroundWarning); }
.crit { background: var(--backgroundCritical); }
</style>

<h4>About Abelbeck Aviation Checklist</h4>

<h5>Overview</h5>
<p>
	This web application shows checklists. You can check each item, and sometimes
	you can enter values like fuel volumes, or times. If you have checked all items
	on a list, the logical next one (as defined in the checklist HTML file) is presented.
</p>
<p>
	At the top, several fields for typical values announced over radio are displayed.
	You can click on them to enter a new value or delete the current one.
</p>
<p>
	The list of alredy-checked lists, the theme mode, and any user-modified variable are saved in the URL.
	Intermediate checklist states are not preserved due to safety reasons (suspending and resuming checklists is error-prone).
</p>

<h5>Navigation</h5>
<p>
	If loaded without query parameters (=URL string after the first question mark), an overview of all known checklists is shown.
	The checklists are announced by an entry in the index.json file in the checklist directory.
	If at least the callsign query is present ("?.callsign=CALLSIGN"), the checklists for that callsign are shown (if they exist).
</p>
<p>
	With a click on the checklist's heading you get an overview of all similar checklists (either normal or emergency procedures).
	With a click on an overview's heading (either normal or emergency procedures) you are taken back to the currently selected checklist.
</p>
<p>
	The footer buttons offer the following functions (left to right):
</p>
<ul>
	<li><samp class=ok>&blacktriangleleft;</samp>: go to logical previous checklist;</li>
	<li><samp class=warn>!&Congruent;</samp>: go to first unfinished checklist;</li>
	<li><samp>&blacktriangleright;&Congruent;&blacktriangleleft;</samp>: go to currently selected checklist;</li>
	<li><samp>&vellip;</samp>: go to the menu;</li>
	<li><samp class=crit> </samp>: go to the emergency procedures list;</li>
	<li><samp class=ok>&blacktriangleright;</samp>: go to logical next checklist.</li>
</ul>
<p>
	The menu offers the following options:
</p>
<ul>
	<li>Load a previously saved checklist state from a local file;</li>
	<li>Save the current checklist state to a local file;</li>
	<li>Clear the current checklist (i.e. start anew);</li>
	<li>Show this information screen;</li>
	<li>Toggle between dark and light theme;</li>
	<li>Exit and return to the main app screen.</li>
</ul>

<h5>Licence: Graphic Files</h5>
<p>
	All graphic files in the <code>media/</code> subdirectory are provided under the terms and conditions of the
	<strong>Create Commons BY-NC-SA 4.0 license</strong>.
	You are free to <strong>share</strong> and <strong>adapt</strong> the material under the following terms:
	<ul>
		<li><strong>Attribution (BY)</strong> &ndash;&nbsp;You must give appropriate credit;</li>
		<li><strong>NonCommercial (NC)</strong> &ndash;&nbsp;You may not use the material for commercial purposes;</li>
		<li><strong>ShareAlike (SA)</strong> &ndash;&nbsp;You have to distribute your modified material unter the same licence.</li>
	</ul>
</p>
<p>
	For further details visit <a href="https://creativecommons.org/licenses/by-nc-sa/4.0/">the creativecommons.org license page</a>.
</p>

<h5>Licence: Software</h5>
<p>
	Copyright 2024 <a href="mailto:frank@abelbeck.info">Frank Abelbeck</a>
</p>
<p>
	The Abelbeck Aviation Checklist (AAC) toolbox is free software: you can redistribute it and/or modify it under
	the terms of the <strong>GNU General Public License</strong> as published by the Free Software
	Foundation, either <strong>version 3</strong> of the License, or (at your option) any later version.
</p>
<p>
	The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
	FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
</p>
<p>
	An HTML version of the GNU General Public License 3 text is available <a href="./COPYING.html" target=_blank>on this website</a>
	or <a href="http://www.gnu.org/licenses">on the official gnu.org site</a>.
</p>
`;

/*
 * default initial warning message
 */
let strInitialWarning = `
<p>
	The Abelbeck Aviation Checklist (AAC) toolbox is meant as an additional, <em>INOFFICIAL</em> tool.
	It <em>DOES NOT</em> replace or override the official checklists and the flight manual accompaning your aircraft.
	All official procedures take precedence in any case and it is your responsibility to comply with them.
</p>
<p>
	The AAC toolbox is distributed in the hope that it will be useful, but <em>WITHOUT ANY WARRANTY</em>;
	without even the implied warranty of <em>MERCHANTABILITY</em> or <em>FITNESS FOR A PARTICULAR PURPOSE</em>.
</p>
`;

export class ChecklistManager extends AacWebDocumentManager {
	
	constructor(mapMeta) {
		super(mapMeta);
		this._mapIndex = new AsyncIOMap();
		this._mapNodes = new AsyncIOMap();
		this._nodeMain = null;
		this._varManager = new VariableManager();
		this._keyPadManager = new KeyPadManager();
		/*
		 * state information, to configure a checklist only based on URI information
		 *  - callsign -------------------> this._urlState.searchParams(".callsign")
		 *  - current checklist ----------> this._urlState.hash
		 *  - list of checked sections ---> this._urlState.searchParams(".checked")
		 *  - dark mode ------------------> this._urlState.searchParams(".dark")
		 *  - variable name:value --------> this._urlState.searchParams(any other variable name)
		 * 
		 * associate a title with a variable --> set keypad title from it (_varManager.getDescription(strName))
		 * 
		 */
	}
	
	async runMain() {
		await this.buildChecklistIndex();
	}
	
	async buildChecklistIndex() {
		let objIndex = await this.readJson("./checklists/index.json");
		let headerChecklist, headerIcon;
		for (let [strCallsign,objChecklist] of Object.entries(objIndex)) {
			if (objChecklist.file !== undefined) {
				/*
				 * check if file and icon exist by requesting headers
				 * non-existent checklists are skipped, non-existent icons are disabled
				 */
				try {
					headerChecklist = await this.readHeader(objChecklist.file);
				} catch (err) {
					console.warn(`checklist file '${objChecklist.file}' not found, skipping`);
					continue;
				}
				try {
					headerIcon = await this.readHeader(objChecklist.icon);
				} catch (err) {
					console.warn(`icon file '${objChecklist.icon}' not found`)
					objChecklist.icon = undefined;
				}
				await this._mapIndex.set(
					strCallsign,
					[
						objChecklist.file,
						objChecklist.type,
						objChecklist.icon,
						(new Date(headerChecklist.get("last-modified"))).valueOf(),
						Number(headerChecklist.get("content-length"))
					]
				);
			}
		}
	}
	
	async updateStateMain() {
		/*
		 * Update the document, based on the current URL. This callback method 
		 * is triggered by popstate events
		 */
		let strCallsign = this._urlState.searchParams.get(".callsign");
		if (strCallsign && strCallsign.length > 0) {
			/*
			 * show checklist for given callsign
			 */
			await this.loadChecklist(strCallsign);
			await this.configureChecklist();
			
		} else {
			/*
			 * no callsign given: show main menu, ignore queries
			 */
			await this.buildChecklistIndex();
			await this.showMainMenu();
		}
		super.updateStateMain();
	}
	
	async configureChecklist() {
		/*
		 * configure the checklist, based on infos in the URL
		 * 
		 * query parameter     meaning
		 * ---------------------------------------------------------------------
		 * .callsign           checklist identifier, already processed
		 * .checked            space-separated list of section ids
		 * .dark               true or false, theme setting
		 * <any identifier>    value of a variable
		 * 
		 * fragment: currently shown checklist
		 */
		let strUrlFragment = this._urlState.hash;
		for (const [strQuery,strValue] of this._urlState.searchParams) {
			switch (strQuery) {
				case ".checked":
					let setChecked = new Set( this._urlState.searchParams.get(".checked").split(" ") );
					for (const node of this._nodeMain.querySelectorAll("section")) {
						if (setChecked.has(node.id)) {
							this.checkSection(node);
						}
					}
					break;
				case ".dark":
					document.documentElement.classList.add("dark");
					break;
				default:
					if (strQuery[0] !== ".") {
						await this._varManager.setValue(strQuery,strValue);
						for (const node of document.querySelectorAll(`section [data-input="${strQuery}"]`)) {
							node.setAttribute("checked","true");
						}
					}
			}
		}
		this.showScreen(strUrlFragment);
	}
	
	/**
	 * Read given file as HTML and return the resulting HTML document.
	 * 
	 * If the first main element defines a data-inherit attribute, this function
	 * recurses into the given file to pre-populate the HTML document. In this
	 * case the definitions if this file are added to the pre-populated
	 * document, based on the following rules:
	 * 
	 *  - main/style: append after existing elements.
	 *  - main/dfn: append after existing elements, replace dfns with same data-name.
	 *  - main/var: append after existing elements, replace dfns with same data-name.
	 *  - main/article: append sections to corresponding article (data-emergency or not) after existing sections; replace sections with same id.
	 * 
	 * Processed files are passed as second argument to detect and avoid loops.
	 * 
	 * Any errors are thrown upwards.
	 */
	async readChecklistHtml(strFile,setFilesVisited) {
		// console.log(`readChecklistHtml(${strFile},${setFilesVisited})`);
		if (setFilesVisited === undefined) {
			setFilesVisited = new Set();
		}
		let docHtml, nodeRef;
		let docHtmlThis = await this.readHtml(strFile);
		let strFileParent;
		try {
			strFileParent = docHtmlThis.querySelector("main").getAttribute("data-inherit");
		} catch (err) {
			console.warn(err);
		}
		if (strFileParent) {
			// check for loops, break recursion
			if (setFilesVisited.has(strFileParent)) {
				console.warn(`checklist inheritance loop detected, ignoring file '${strFileParent}'`);
				docHtml = docHtmlThis;
				
			} else {
				// record files in set of files and recurse
				setFilesVisited.add(strFileParent);
				docHtml = await this.readChecklistHtml(strFileParent,setFilesVisited);
				
				/*
				 * import all sections into the corresponding articles (data-emergency?),
				 * but let current document's versions take precedence (id attribute)
				 * 
				 * first get articles and create them if they are missing
				 */
				let nodeMain = docHtml.querySelector("main");
				if (!nodeMain) {
					nodeMain = docHtml.body.appendChild(AacWebDocumentManager.constructNode("main"));
				}
				
				let nodeNormal = docHtml.querySelector("main > article:not([data-emergency])");
				if (!nodeNormal) {
					nodeNormal = nodeMain.appendChild( AacWebDocumentManager.constructNode("article"));
				}
				
				let nodeEmergency = docHtml.querySelector("main > article[data-emergency]");
				if (!nodeEmergency) {
					nodeEmergency = nodeMain.appendChild( AacWebDocumentManager.constructNode("article",undefined,{ "data-emergency":"" }) );
				}
				
				for (let nodeSection of docHtmlThis.querySelectorAll("main > article > section")) {
					try {
						docHtml.getElementById(nodeSection.id).replaceWith(nodeSection)
					} catch (err) {
						if (nodeSection.parentNode.dataset.emergency !== undefined) {
							nodeEmergency.appendChild(nodeSection);
						} else {
							nodeNormal.appendChild(nodeSection);
						}
					}
				}
				
				/*
				 * get last main style/dfn/var element of the parent document and
				 * insert any main styles/dfns/vars of the current document after it
				 * 
				 * intercept any douplicate element (same data-name) and
				 * let element in the current document take precedence
				 */
				for (let strTag of ["style","dfn","var"]) {
					nodeRef = docHtml.querySelector(`main > ${strTag}:last-of-type`);
					if (!nodeRef) {
						/*
						 * no style|dfn|var element found: just insert before first article
						 * as we have imported articles and created article nodes before (see above),
						 * this will always return a node
						 */
						nodeRef = docHtml.querySelector("main > article:first-of-type");
					}
					for (let node of docHtmlThis.querySelectorAll(`main > ${strTag}`)) {
						try {
							docHtml.querySelector(`${strTag}[data-name=${node.dataset.name}]`).replaceWith(node);
						} catch (err) {
							nodeRef.after(node);
						}
					}
				}
			}
		} else {
			docHtml = docHtmlThis;
		}
		return docHtml;
	}
	
	/**
	 * If no main node with the id=strCallsign exists, load the checklist file
	 * associated with given callsign (see _mapIndex) and configure it according
	 * to the queries (checked sections, variables) and fragment (current
	 * section/menu to show).
	 * 
	 * If the main node exists, just return.
	 */
	async loadChecklist(strCallsign) {
		let node, nodeHeader, nodeFooter, nodeListNormal, nodeListEmergency;
		let docHtml;
		if (this._nodeMain && this._nodeMain.id === strCallsign || !(await this._mapIndex.has(strCallsign))) return;
		
		/*
		 * main node for this callsign not found: load it
		 * 
		 *  - construct main element and connect it to a screen manager
		 *  - create screens for menu, normal and emergency lists, and info
		 *  - setup header: ATC buttons
		 *  - setup footer: todo, emergency, and menu buttons
		 *  - finally create basic structure with header, main, and footer
		 *  - activate checklist stylesheet
		 */
		let [strFile,strType,strIcon,numLastModified,numContentLength] = await this._mapIndex.get(strCallsign);
		
		if (this._nodeMain) {
			this._nodeMain.id = strCallsign;
		} else {
			this._nodeMain = AacWebDocumentManager.constructNode("main",undefined,{ id:strCallsign });
		}
		
		/*
		 * clear state vars, reset theme, and get URL fragment (=current screen)
		 */
		await this._mapNodes.clear();
		await this._varManager.clear();
		await this._keyPadManager.clear();
		document.documentElement.classList.remove("dark");
		let strFragment = this._urlState.hash.replace("#","");
		
		/*
		 * construct predefined screens
		 */
		node = AacWebDocumentManager.constructNode("article",null, { id:"aacMenu", hidden:"hidden" });
		node.innerHTML = strHtmlMenu
		this._nodeMain.appendChild(node);
		
		node = AacWebDocumentManager.constructNode("article",null, { id:"aacNormal", hidden:"hidden" });
		node.innerHTML = strHtmlNormalList;
		nodeListNormal = node.querySelector("#listNormal");
		this._nodeMain.appendChild(node);
		
		node = AacWebDocumentManager.constructNode("article",null, { id:"aacEmergency", hidden:"hidden" });
		node.innerHTML = strHtmlEmergencyList;
		nodeListEmergency = node.querySelector("#listEmergency");
		this._nodeMain.appendChild(node);
		
		node = AacWebDocumentManager.constructNode("article",null, { id:"aacInfo", hidden:"hidden" });
		node.innerHTML = strHtmlInfo;
		this._nodeMain.appendChild(node);
		
		nodeHeader = AacWebDocumentManager.constructNode("header");
		nodeHeader.innerHTML = strHtmlHeader;
		
		nodeFooter = AacWebDocumentManager.constructNode("footer");
		nodeFooter.innerHTML = strHtmlFooter;
		
		/*
		 * replace main menu style sheet with checklist stylesheet
		 * remove any other stylesheet
		 */
		document.body.replaceChildren(nodeHeader,this._nodeMain,nodeFooter);
		for (const nodeStyle of document.head.querySelectorAll("link[rel=stylesheet]")) {
			if (nodeStyle.id === "aacMainStyleSheet") {
				nodeStyle.setAttribute("href","css/checklist.css");
			} else {
				nodeStyle.remove();
			}
		}
		
		/*
		 * load checklist file as HTML document
		 * 
		 * 2024-12-31: moved to separate function in order to implement inheritance
		 */
		try {
			docHtml = await this.readChecklistHtml(strFile);
		} catch (err) {
			console.warn(`failed to read checklist file '${strFile}': ${err}`);
			history.back();
			return;
		}
		
		/*
		 * process extra style definitions
		 */
		nodeHeader.before(...docHtml.querySelectorAll(":not(section) > style"));
		
		/*
		 * process type definitions
		 */
		let mapTypes = new Map();
		for (const node of docHtml.querySelectorAll("main > dfn[data-name][data-type][data-keypad][data-format]")) {
			mapTypes.set(node.dataset.name,[node.dataset.type,node.dataset.keypad,node.dataset.format]);
		}
		
		/*
		 * process variable declarations and add ATC variables 
		 */
		let mapInitialValues = new Map();
		let mapExpressions = new Map();
		for (const node of docHtml.querySelectorAll("main > var[data-name][data-type]")) {
			try {
				/*
				 * check if the user defined a custom type by looking up strType in mapTypes
				 * if not, use the type passed to the var element (might fail if not defined)
				 * 
				 * get keypad identifier either from type definition or from the VariableManager
				 * get format string either from type definition or from the data-format attribute
				 */
				let strType,strKeyPad,strFormat;
				if (mapTypes.has(node.dataset.type)) {
					// prefer custom type over basic types
					[strType,strKeyPad,strFormat] = mapTypes.get(node.dataset.type);
				} else {
					strType = node.dataset.type;
					strKeyPad = node.dataset.keypad;
					if (strKeyPad === undefined && VariableManager.mapTypes.has(strType)) strKeyPad = strType;
					strFormat = node.dataset.format;
				}
				/*
				 * add to variable manager
				 * add to keypad manager
				 * memorise any expression and initial value (to be applied when everything is set up)
				 */
				await this._varManager.add( node.dataset.name, strType, node.dataset.description, strFormat, node.dataset.expr);
				await this._keyPadManager.add(node.dataset.name,strKeyPad);
				if (node.innerHTML.trim() !== "") {
					mapInitialValues.set(node.dataset.name,node.innerHTML.trim());
				}
				if (node.dataset.expr !== undefined) {
					mapExpressions.set(node.dataset.name,node.dataset.expr);
				}
			} catch (err) {
				console.warn(`ignoring declaration for variable '${node.dataset.name}': ${err}`);
			}
		}
		/*
		 * deal with internal ATC variables defined in the header section
		 */
		for (const node of document.body.querySelectorAll("header > button[data-input][data-keypad][data-prefix]")) {
			await this._varManager.add(node.dataset.input,"string",node.dataset.prefix);
			mapInitialValues.set(node.dataset.input,"");
			await this._varManager.subscribe(node.dataset.input, async (variable) => { node.innerHTML = await variable.toString(); });
			await this._keyPadManager.add(node.dataset.input,node.dataset.keypad);
		}
		
		for (const [strName,strExpression] of mapExpressions) {
			await this._varManager.setExpression(strName,strExpression);
		}
		
		/*
		 * process sections
		 */
		let intIdx = 0;
		let nodePrev,nodeNext;
		for (const nodeSection of docHtml.querySelectorAll("main > article > section")) {
			/*
			 * ensure that every section as an id and that a h4 heading is present
			 * determine type (emergency, nocheck, skip)
			 */ 
			if (nodeSection.id === "") {
				nodeSection.id = `${strCallsign}${intIdx++}`;
			}
			let nodeH4 = nodeSection.querySelector(":scope > h4");
			if (nodeH4 === null) {
				console.warn(`section id='${nodeSection.id}' without heading, skipping`);
				continue;
			}
			let boolIsEmergency = nodeSection.parentNode.dataset.emergency !== undefined;
			
			if (boolIsEmergency) nodeSection.dataset.emergency = "";
			
			if (nodeSection.id !== strFragment) nodeSection.setAttribute("hidden","hidden");
			
			let boolNoCheck = "nocheck" in nodeSection.dataset || boolIsEmergency;
			let boolSkip = "skip" in nodeSection.dataset;
			
			if (nodeSection.dataset.reveal !== undefined) {
				/*
				 * "reveal" section: hide all definition lists (elements with data-show will unhide them)
				 */
				for (const nodeDl of nodeSection.querySelectorAll("dl")) {
					nodeDl.setAttribute("hidden","hidden");
				}
			}
			
			/*
			 * record section in lists
			 */
			let nodeLi = AacWebDocumentManager.constructNode("li",undefined,{ "data-goto":nodeSection.id, class:"aacActiveElement" });
			if (boolNoCheck) nodeLi.classList.add("aacNoCheck");
			nodeLi.innerHTML = nodeH4.innerHTML;
			if (boolIsEmergency) {
				nodeListEmergency.appendChild(nodeLi);
			} else {
				if (nodeSection.id === strFragment) nodeLi.classList.add("aacCurrentSection");
				nodeListNormal.appendChild(nodeLi);
			}
			
			let nodeSpan  = AacWebDocumentManager.constructNode("span",undefined,{ "data-list": (boolIsEmergency) ? "emergency" : "normal" });
			nodeSpan.innerHTML = nodeH4.innerHTML;
			nodeH4.replaceChildren(nodeSpan);
			if (!boolNoCheck) {
				/*
				 * checkable section: wrap heading text in a span and add an anchor
				 */
				let nodeReset = AacWebDocumentManager.constructNode("a","&Cross;",{ "data-reset": nodeSection.id });
				let nodeSkip  = AacWebDocumentManager.constructNode("a","&check;",{ "data-skip": nodeSection.id });
				nodeH4.append(nodeSkip,nodeReset);
			} else {
			}
			
			for (const nodeDt of nodeSection.querySelectorAll("dt")) {
				let nodeDd = nodeDt.nextElementSibling;
				/*
				 * since dt is not easy to address from its next sibling dd in CSS,
				 * plus keeping both in sync, we wrap dt and dd in either a span (output, nocheck)
				 * or an anchor (input, check items)
				 */
				let nodeWrap;
				if (nodeDd.dataset.output !== undefined || nodeDd.dataset.nocheck !== undefined || boolNoCheck) {
					nodeWrap = AacWebDocumentManager.constructNode("span");
				} else {
					nodeWrap = AacWebDocumentManager.constructNode("a");
				}
				nodeDt.parentNode.insertBefore(nodeWrap,nodeDt);
				nodeWrap.appendChild(nodeDt);
				nodeWrap.appendChild(nodeDd);
				for (const strClass of nodeDt.classList) {
					switch (strClass) {
						case "pilot":
						case "copilot":
						case "central":
						case "warning":
							// transfer dt class to parent
							nodeWrap.classList.add(strClass);
							nodeDt.classList.remove(strClass);
							break;
					}
				}
				
				if (nodeDd.dataset.input !== undefined) {
					// input node: transfer data-input to wrapper, mark with aacInput class
					nodeWrap.classList.add("aacInput");
					nodeWrap.dataset.input = nodeDd.dataset.input;
					nodeDd.dataset.output = nodeDd.dataset.input;
					delete nodeDd.dataset.input;
					
				} else if (nodeDd.dataset.output !== undefined) {
					// output node: mark with aacOutput class
					nodeWrap.classList.add("aacOutput");
					
				} else if (!boolNoCheck) {
					// otherwise, if not in a no-check section: check node
					// add data-check attribute
					nodeWrap.dataset.check = true;
				}
			}
			
			/*
			 * set data-prev to previous node if not yet defined (only applies to non-emergency procedures)
			 * set data-next of previous node to this node if not yet defined (only applies to non-emergency procedures)
			 * make this node the new previous node
			 * 
			 * in the end, append section to main element
			 */
			if (nodeSection.dataset.prev === undefined && nodePrev && nodePrev.dataset.emergency === undefined && nodeSection.dataset.emergency === undefined) {
				nodeSection.setAttribute("data-prev",nodePrev.id);
			}
			if (nodePrev && nodeSection.dataset.emergency === undefined && nodePrev.dataset.next === undefined) {
				nodePrev.setAttribute("data-next",nodeSection.id);
			}
			this._nodeMain.appendChild(nodeSection);
			nodePrev = nodeSection;
		}
			
		/*
		 * check if emergency list is empty; in that case hide the emergency list button
		 */
		if (nodeListNormal.querySelectorAll("li").length <= 0) {
			document.getElementById("btnEmergency").setAttribute("disabled","true");
		}
		
		/*
		 * connect all inputs
		 */
		for (const nodeAction of document.querySelectorAll("[data-input],[data-check],[data-goto],[data-show],[data-exec],[data-reset],[data-list],[data-skip]")) {
			let boolError = false;
			/*
			 * bare minimum an checks: do targets exist? do execs compile?
			 */
			if (nodeAction.dataset.show !== undefined) {
				let nodeToShow = document.getElementById(nodeAction.dataset.show);
				if (nodeToShow !== null) {
					if (nodeAction.id.length <= 0) {
						nodeAction.id = nodeAction.dataset.show + "_backlink"
					}
					nodeToShow.dataset.showbacklink = nodeAction.id
				} else {
					console.warn(`data-show deactivated: node ${nodeAction} references unknown id '${nodeAction.dataset.show}'`);
					boolError = true;
				}
			}
			
			if (nodeAction.dataset.goto !== undefined) {
				let nodeGoTo = document.getElementById(nodeAction.dataset.goto);
				if (nodeGoTo === null) {
					console.warn(`data-goto deactivated: node ${nodeAction} references unknown id '${nodeAction.dataset.goto}'`);
					boolError = true;
				} else if (nodeGoTo.tagName != "SECTION") {
					console.warn(`data-goto deactivated: node ${nodeAction} references non-section id '${nodeAction.dataset.goto}'`);
					boolError = true;
				}
			}
			
			if (nodeAction.dataset.input !== undefined && !this._varManager.has(nodeAction.dataset.input)) {
				console.warn(`data-input deactivated: node ${nodeAction} references undefined variable '${nodeAction.dataset.input}'`);
				boolError = true;
			}
			
			if (nodeAction.dataset.reset !== undefined) {
				let nodeToReset = document.getElementById(nodeAction.dataset.reset);
				if (nodeToReset === null) {
					console.warn(`data-reset deactivated: node ${nodeAction} references unknown id '${nodeAction.dataset.reset}'`);
					boolError = true;
				} else if (nodeToReset.tagName != "SECTION") {
					console.warn(`data-goto deactivated: node ${nodeAction} references non-section id '${nodeAction.dataset.reset}'`);
					boolError = true;
				}
			}
			
			if (nodeAction.dataset.list !== undefined && nodeAction.dataset.list !== "normal" && nodeAction.dataset.list !== "emergency") {
				console.warn(`data-list deactivated: node ${nodeAction} references unknown list '${nodeAction.dataset.list}'`);
				boolError = true;
			}
			
			if (!boolError) {
				nodeAction.addEventListener("click", async () => { await this.processClickEvent(nodeAction); });
			}
		}
		
		for (const nodeOutput of document.querySelectorAll("[data-output]")) {
			await this._varManager.subscribe(nodeOutput.dataset.output, async (variable) => {
				try {
					nodeOutput.innerHTML = await variable.toString();
				} catch (err) {
					console.warn(`failed to set output node <${nodeOutput.tagName}>: ${err}`);
				}
			});
		}
		
		/*
		 * connect all buttons with an id
		 */
		for (const nodeButton of document.querySelectorAll("button[id]")) {
			nodeButton.addEventListener("click", async () => { await this.processClickEvent(nodeButton); });
		}
		
		/*
		 * deal with dynamic attributes
		 * they are given with data-dynattr="name1=variableName1; name2=variableName2; ..."
		 * if a variable changes, the named attribute in this node is updated
		 * 
		 * mapping: variable -> nodes -> attributes
		 */
		let mapDynAttr = new Map();
		for (const nodeDynAttr of document.querySelectorAll("[data-dynattr]")) {
			for (const [strAttrName,strVariableName] of new Map( // map attribute name to expression
				nodeDynAttr.dataset.dynattr
					.split(";") // split string along ';'
					.filter( (val) => val.trim().length > 0) // ignore any substring which is empty after being trimmed
					.map( (val) => val.split("=",2) // split substring at first "=" into attribute name and variable identifier
						.map( (val) => val.trim() ) // trim values
					)
			)) {
				let mapDynAttrNodes = mapDynAttr.get(strVariableName);
				if (mapDynAttrNodes === undefined) {
					mapDynAttrNodes = new Map();
					mapDynAttr.set(strVariableName, mapDynAttrNodes);
				}
				let setDynAttrAttributes = mapDynAttrNodes.get(nodeDynAttr);
				if (setDynAttrAttributes === undefined) {
					setDynAttrAttributes = new Set();
					mapDynAttrNodes.set(nodeDynAttr,setDynAttrAttributes);
				}
				setDynAttrAttributes.add(strAttrName);
			}
		}
		for (const [strVariableName,mapDynAttrNodes] of mapDynAttr) {
			this._varManager.subscribe(strVariableName, async (variable) => {
				let value;
				try {
					for (const [nodeDynAttr,setDynAttrAttributes] of mapDynAttrNodes) {
						for (const strAttrName of setDynAttrAttributes) {
							if (strAttrName[0] === "$") {
								// variable name starts with a dollar sign: get string representation
								value = await variable.toString();
								strAttrName.slice(1);
							} else {
								// otherwise, get variable value
								value = await variable.getValue();
							}
							if (value === null || value === undefined || typeof(value) === "number" && isNaN(value)) {
								// attribute values are automatically converted to strings,
								// so in case of null or undefined or NaN just remove the attribute
								nodeDynAttr.removeAttribute(strAttrName);
							} else {
								nodeDynAttr.setAttribute(strAttrName, value);
							}
						}
					}
				} catch (err) {
					console.warn(`failed to update dynamic attribute '${nodeDynAttr.dataset.dynattr}': ${err}`);
				}
			});
		}
		
		/*
		 * install keypad nodes with attribute hidden=hidden
		 * compile all auto-expressions
		 * set all variable values (triggering updates to dependents)
		 */
		this._keyPadManager.installNodes(this._nodeMain,{ hidden:"hidden" });
		for (const [strName,strValue] of mapInitialValues) {
			await this._varManager.setInitialValue(strName,strValue);
			await this._varManager.setValue(strName,strValue);
		}
		
		/*
		 * finally: show a warning message to the user to not rely solely on this app
		 */
		let nodeDiv = AacWebDocumentManager.constructNode("div");
		let nodeDialog = AacWebDocumentManager.constructNode("dialog", [
			nodeDiv,
			AacWebDocumentManager.constructButton("I read and understood this message!", (event) => { nodeDialog.close(); nodeDialog.remove(); })
		]);
		let strUserWarning = await this._varManager.getValue("_strAacInitialWarning");
		if (strUserWarning === undefined) strUserWarning = "";
		nodeDiv.innerHTML = strInitialWarning + strUserWarning;
		
		document.body.appendChild(nodeDialog);
		nodeDialog.showModal();
	}
	
	showScreen(strName, boolUpdateUrl=true) {
		if (strName === undefined || strName === "") {
			let nodeFirst = this._nodeMain.querySelector("section:not([data-emergency],[data-nocheck],[checked])");
			if (nodeFirst === null) {
				nodeFirst = this._nodeMain.querySelector("section");
				if (nodeFirst === null) {
					return;
				}
			}
			strName = nodeFirst.id;
		}
		if (boolUpdateUrl) {
			this._urlState.hash = strName;
		}
		
		strName = strName.replace("#","");
		for (const node of this._nodeMain.childNodes) {
			if (node.id === strName) {
				/*
				 * show node matching the given name
				 * update prev/next buttons (data-goto will be automatically parsed by processClickEvent)
				 */
				node.removeAttribute("hidden");
				
				let nodeBtnPrev = document.getElementById("btnPrev");
				if (node.dataset.prev !== undefined) {
					nodeBtnPrev.dataset.goto = node.dataset.prev;
					nodeBtnPrev.removeAttribute("disabled");
				} else {
					delete nodeBtnPrev.dataset.goto;
					nodeBtnPrev.setAttribute("disabled","true");
				}
				
				let nodeBtnNext = document.getElementById("btnNext");
				if (node.dataset.next !== undefined) {
					nodeBtnNext.dataset.goto = node.dataset.next;
					nodeBtnNext.removeAttribute("disabled");
				} else {
					delete nodeBtnNext.dataset.goto;
					nodeBtnNext.setAttribute("disabled","true");
				}
				
				if (node.dataset.emergency !== undefined) {
					document.body.classList.add("aacEmergency");
				} else {
					document.body.classList.remove("aacEmergency");
				}
			} else {
				/*
				 * hide any node not matching the given name
				 */
				node.setAttribute("hidden","hidden");
			}
		}
		if (boolUpdateUrl) {
			/*
			 * if the URL fragment should be updated, modify the normal list selection
			 */
			for (const node of this._nodeMain.querySelectorAll("#listNormal li[data-goto]")) {
				if (node.dataset.goto === strName) {
					node.classList.add("aacCurrentSection");
				} else {
					node.classList.remove("aacCurrentSection");
				}
			}
		}
	}
	
	isShown(...arrStrName) {
		let boolShown = false;
		let node;
		for (const strName of arrStrName) {
			node = document.getElementById(strName);
			boolShown ||= (node && !node.hasAttribute("hidden"));
		}
		return boolShown;
	}
	
	showScreenToggle(strToShow,boolUpdateUrl,strFallback,boolUpdateUrlFallback) {
		if (this.isShown(strToShow)) {
			// already shown: return to fallback
			this.showScreen(strFallback,boolUpdateUrlFallback);
		} else {
			this.showScreen(strToShow,boolUpdateUrl);
		}
	}
	
	async showMainMenu() {
		/*
		 * construct main menu:
		 * 
		 *  - create nav element and populate it with anchors to the checklists
		 *  - create simple page with header, main, and footer and replace the current document body
		 *  - set main stylesheet 
		 */
		let nodeNav = AacWebDocumentManager.constructNode("nav",[
			AacWebDocumentManager.constructNode("a","Back to Main Menu", { href:"./index.html" } ),
			AacWebDocumentManager.constructButton("Load...", async () => {
				try {
					navigator.vibrate(100);
				} catch (err) {}
				this.setUrlState( JSON.parse( await (await this.loadFile()).text() ) );
			}),
		]);
		
		let nodeA;
		await this._mapIndex.forEach( async (strCallsign,[strFile,strType,strIcon,numLastModified,numContentLength]) => {
			nodeA = AacWebDocumentManager.constructNode("a",
				AacWebDocumentManager.constructNode("span",`${strCallsign} (${strType})`),
				{ href:`?.callsign=${strCallsign}` }
			);
			if (strIcon) {
				nodeA.appendChild(AacWebDocumentManager.constructNode("img","",{ src:strIcon}));
			}
			nodeNav.appendChild(nodeA);
		});
		document.body.replaceChildren(
			this.constructStandardHeader("Abelbeck Aviation Checklist"),
			AacWebDocumentManager.constructNode("main",nodeNav),
			this.constructStandardFooter()
		);
		document.head.querySelector("link[rel=stylesheet]").setAttribute("href","css/main.css");
	}
	
	async processClickEvent(node) {
		/*
		 * acknowledge event and send haptic feedback
		 */
		try {
			navigator.vibrate(100);
		} catch (err) {}
		
		let nodeSection;
		
		if (node.dataset.show !== undefined) {
			/*
			 * user clicked a data-show element:
			 * reveal the node with id=dataset.show
			 * hide all sibling nodes with the same tagname
			 */
			let nodeToShow = document.getElementById(node.dataset.show);
			for (const nodeChild of nodeToShow.parentNode.querySelectorAll(`${nodeToShow.tagName}:not([hidden])`)) {
				nodeChild.setAttribute("hidden","hidden");
			}
			document.getElementById(node.dataset.show).removeAttribute("hidden");
			/*
			 * remove aacShowHighlight from all but this node
			 */
			nodeSection = document.querySelector(this._urlState.hash);
			for (const nodeChild of nodeSection.querySelectorAll("[data-show]")) {
				nodeChild.classList.remove("aacShowHighlight");
			}
			node.classList.add("aacShowHighlight");
		}
		
		if (node.dataset.check !== undefined) {
			/*
			 * user clicked a data-check element: set checked attribute of this element
			 */
			node.setAttribute("checked","true");
			this.checkHierarchy(node);
		}
		
		if (node.dataset.exec !== undefined && node.dataset.execStore !== undefined) {
			/*
			 * user clicked a data-exec element: calculate and update variable given as node.dataset.execStore 
			 */
			if (await this._varManager.has(node.dataset.execStore)) {
				try {
					let boolChanged = await this._varManager.setValue(
						node.dataset.execStore,
						await this._varManager.execute(node.dataset.exec)
					);
					if (boolChanged) {
						this._urlState.searchParams.set(node.dataset.execStore, await this._varManager.getValue(node.dataset.execStore));
					}
				} catch (err) {
					console.warn(`failed to calculate data-exec='${node.dataset.exec}': ${err}`);
				}
			}
		}
		
		if (node.dataset.input !== undefined) {
			/*
			 * user clicked an input element: dataset.input=variable name 
			 *  - retrieve keypad for that variable from the keypad manager
			 *  - set title from variable description
			 *  - set value (retrieve value from variable manager)
			 *  - save current screen in order to return to it when closing the keypad
			 *  - install callbacks for enter and escape events
			 *  - update URI to save title and value
			 *  - show keypad
			 */
			let keypad = await this._keyPadManager.get(node.dataset.input);
			keypad.setTitle(await this._varManager.getDescription(node.dataset.input));
			let [strPrefix,strSuffix] = await this._varManager.getPrefixSuffix(node.dataset.input);
			keypad.setPrefix(strPrefix);
			keypad.setSuffix(strSuffix);
			keypad.setValue(await this._varManager.getValue(node.dataset.input));
			keypad.setCallbackEnter( async (value) => {
				/*
				 * user clicked Enter:
				 *  - set variable value
				 *  - update url if value changed
				 *  - head back to previous screen
				 */
				let boolChanged = await this._varManager.setValue(node.dataset.input,value);
				if (boolChanged) {
					if (await this._varManager.isEmpty(node.dataset.input)) {
						// empty vars are removed from the URL
						this._urlState.searchParams.delete(node.dataset.input);
					} else {
						this._urlState.searchParams.set(node.dataset.input, await this._varManager.getValue(node.dataset.input));
					}
				}
				this.showScreen(this._urlState.hash,false);
				node.setAttribute("checked","true");
				this.checkHierarchy(node);
				this.updateUrl();
			});
			keypad.setCallbackEscape( async () => {
				/*
				 * user clicked Escape: head back to previous screen
				 */
				this.showScreen(this._urlState.hash,false);
				this.updateUrl();
			});
			
			this.showScreen(keypad.getNode().id,false); // show screen, but don't record in URL
		}
		
		if (node.dataset.goto !== undefined) {
			/*
			 * user clicked a goto element: change state to show section id=dataset.goto
			 */
			this.showScreen(node.dataset.goto);
		}
		
		if (node.dataset.reset !== undefined) {
			/*
			 * user clicked a reset element: reset section with id=dataset.reset
			 */
			await this.resetSection(document.getElementById(node.dataset.reset));
		}
		
		if (node.dataset.skip !== undefined) {
			/*
			 * user clicked a skip element: check all items in the current section
			 */
			this.checkSection( document.getElementById(node.dataset.skip) );
		}
		
		switch (node.dataset.list) {
			case "emergency":
				this.showScreenToggle("aacEmergency",false,this._urlState.hash,true);
				break;
			case "normal":
				this.showScreenToggle("aacNormal",false,this._urlState.hash,true);
				break;
		}
		
		switch (node.id) {
			case "btnEmergency": // open emergency list
				this.showScreenToggle("aacEmergency",false,this._urlState.hash,true);
				break;
				
			case "btnToDo": // move to most recent to-do checklist
				this.showScreen();
				break;
				
			case "btnCurrent": // move to currently selected checklist
				this.showScreen(this._urlState.hash);
				break;
				
			case "btnInfo":
				this.showScreen("aacInfo",false);
				break;
				
			case "btnSave": // save data to a file (filepicker via <a download>) and go return to current screen
				let dateNow = new Date();
				this.saveFile(
					`Checklist_${this._nodeMain.id}_${dateNow.toISOString().replace(/[-:]|\.*/g,"")}.json`,
					new Blob([JSON.stringify( this.getUrlState() )], { type:"application/json" })
				);
				this.showScreen(this._urlState.hash);
				break;
				
			case "btnLoad": // load data from a file (filepicker via hidden input) and go to most recent to-do
				await this.setUrlState( JSON.parse( await (await this.loadFile()).text() ) );
				this.showScreen();
				break;
				
			case "btnClear": // clear state, reload
				await this.resetAll();
				break;
				
			case "btnMenu":
				this.showScreenToggle("aacMenu",false,this._urlState.hash,true);
				break;
				
			case "btnTheme":
				this.toggleDarkMode(false);
				break;
				
			case "btnQuit":
				let boolDark = this._urlState.searchParams.has(".dark");
				this._urlState.search = "";
				this._urlState.hash = "";
				if (boolDark) this._urlState.searchParams.set(".dark","");
				document.location = this._urlState;
				return;
		}
		
		this.updateUrl();
	}
	
	addToSetChecked(strName) {
		let setChecked;
		try {
			setChecked = new Set( this._urlState.searchParams.get(".checked").split(" ") );
		} catch (err) {
			setChecked = new Set();
		}
		setChecked.add(strName);
		this._urlState.searchParams.set(".checked",Array.from(setChecked).join(" "));
	}
	
	deleteFromSetChecked(strName) {
		try {
			let setChecked = new Set( this._urlState.searchParams.get(".checked").split(" ") );
			setChecked.delete(strName);
			this._urlState.searchParams.set(".checked",Array.from(setChecked).join(" "));
		} catch (err) {}
	}
	
	getUrlState() {
		let objState = { vars:{} }
		for (const [strKey,strValue] of this._urlState.searchParams) {
			switch (strKey) {
				case ".callsign":
					objState.callsign = strValue;
					break;
				case ".checked":
					objState.checked = this._urlState.searchParams.get(".checked").split(" ");
					break;
				case ".dark":
					objState.dark = true;
					break;
				default:
					objState.vars[strKey] = strValue;
					break;
			}
		}
		let strFragment = this._urlState.hash.replace("#","");
		if (strFragment !== "") {
			objState.current = strFragment;
		}
		return objState;
	}
	
	async setUrlState(objState) {
		if (objState.callsign !== this._nodeMain.id) {
			/*
			 * state object not meant for current checklist:
			 * create new
			 */
			await this.loadChecklist(objState.callsign);
			this.configureChecklist();
		}
		// failed to load new checklist: ignore
		if (objState.callsign !== this._nodeMain.id) return;
		
		this._urlState.search = "";
		this._urlState.searchParams.set(".callsign",objState.callsign);
		if (objState.checked !== undefined) {
			this._urlState.searchParams.set(".checked",objState.checked.join(" "));
		}
		if (objState.dark !== undefined) {
			this._urlState.searchParams.set(".dark", objState.dark);
		}
		for (const [strKey,strValue] of Object.entries(objState.vars)) {
			this._urlState.searchParams.set(strKey,strValue);
		}
		this._urlState.hash = objState.current
		this.updateUrl();
		this.configureChecklist();
	}
	
	checkHierarchy(node) {
		/*
		 * ascend in hierarchy and check parent check status:
		 * if all children with data-check attribute are checked, consider this node checked
		 * if there is a backlink to a data-show node, check this node as well
		 */
		do {
			node = node.parentNode;
			if (node === document.body) break;
			
			/*
			 * calculate progress, i.e. the relation of checked to checkable nodes
			 * 
			 * 2024-12-05: numPercentProgress can be set as data-progress attribute,
			 *             bubbling up to the section and back-linked elements (e.g. list normal),
			 *             might be used to style via attr() function
			 *             problem: if configuring on loading, the exact progress cannot be reconstructed
			 */
			let intNumChecked   = node.querySelectorAll(":is([data-check],[data-input])[checked]").length;
			let intNumCheckable = node.querySelectorAll(":is([data-check],[data-input])").length;
			let numPercentProgress = Math.round(100 * intNumChecked / intNumCheckable);
			
			if (intNumChecked >= intNumCheckable) {
				/*
				 * all checkable children (attribute "data-check") are checked (attribute "checked"):
				 * entire group is checked, so check this node -- if it has no data-nocheck attribute!
				 */
				if (node.dataset.nocheck === undefined) {
					let boolWasChecked = node.hasAttribute("checked");
					node.setAttribute("checked","true");
					
					/*
					 * if there is a back-linked node, check it, too
					 */
					if (node.dataset.showbacklink !== undefined) {
						let nodeBacklink = document.getElementById(node.dataset.showbacklink);
						if (nodeBacklink) {
							nodeBacklink.setAttribute("checked","true");
						}
					}
					
					if (node.tagName == "SECTION") {
						/*
						* reached a section:
						*  - update state in overview list
						*  - update state ("checked" param)
						*  - advance to next section, only if section hasn't been checked before
						*/
						let nodeLi = document.querySelector(`#listNormal li[data-goto="${node.id}"]`);
						if (nodeLi) {
							nodeLi.setAttribute("checked","true");
						}
						this.addToSetChecked(node.id);
						
						/*
						 * unhide goto button for next section
						 */
						let nodeGoTo = node.querySelector("a.aacGotoNext");
						if (nodeGoTo) {
							nodeGoTo.removeAttribute("hidden");
						}
						
						if (!boolWasChecked) {
							this.showScreen(node.dataset.next);
						}
					}
				}
			}
		} while (node.tagName != "SECTION");
	}
	
	checkSection(nodeSection) {
		for (const nodeChild of nodeSection.querySelectorAll("[data-check],[data-input],[data-show]")) {
			nodeChild.setAttribute("checked","true");
			if (nodeChild.dataset.showbacklink !== undefined) {
				/*
				 * if there is a show backlink, check that show node, too.
				 */
				let nodeBacklink = document.getElementById(nodeChild.dataset.showbacklink);
				if (nodeBacklink) {
					nodeBacklink.setAttribute("checked","true");
				}
			}
		}
		
		this.checkHierarchy(nodeSection.firstChild);
	}
	
	async resetSection(nodeSection) {
		/*
		 * un-check section and remove section id from URI checked parameter
		 */
		nodeSection.removeAttribute("checked");
		this.deleteFromSetChecked(nodeSection.id);
		
		/*
		 * uncheck all data-check and data-input children of nodeSection
		 */
		let nodeBacklink;
		for (const nodeChild of nodeSection.querySelectorAll(":is([data-check],[data-input],[data-show])[checked]")) {
			nodeChild.removeAttribute("checked");
			if (nodeChild.dataset.showbacklink !== undefined) {
				/*
				 * if there is a show backlink, uncheck that show node, too.
				 */
				nodeBacklink = document.getElementById(nodeChild.dataset.showbacklink);
				nodeBacklink.removeAttribute("checked");
			}
			if (nodeChild.dataset.input !== undefined) {
				/*
				 * reset variable associated with input element
				 */
				await this._varManager.reset(nodeChild.dataset.input);
			}
		}
		
		/*
		 * hide goto button for next section
		 */
		let nodeGoTo = node.querySelector("a.aacGotoNext");
		if (nodeGoTo) {
			nodeGoTo.setAttribute("hidden","hidden");
		}
		
		/*
		 * uncheck the normal list entry for this nodeSection
		 */
		try {
			nodeBacklink = document.querySelector(`#listNormal li[data-goto=${nodeSection.id}]`);
			nodeBacklink.removeAttribute("checked");
		} catch (err) {}
	}
	
	async resetAll() {
		/*
		 * uncheck all elements
		 * hide all sections
		 * show first section in first article without data-emergency attribute
		 */
		for (const node of this._nodeMain.childNodes) {
			node.setAttribute("hidden","hidden");
		}
		for (const node of this._nodeMain.querySelectorAll("[checked]")) {
			node.removeAttribute("checked");
		}
		
		for (const node of this._nodeMain.querySelectorAll("#listNormal li[data-goto]")) {
			node.classList.remove("aacCurrentSection");
			node.removeAttribute("checked");
		}
		
		await this._varManager.reset();
		
		this._urlState.search = "";
		this._urlState.searchParams.set(".callsign",this._nodeMain.id);
		if (document.documentElement.classList.contains("dark")) {
			this._urlState.searchParams.set(".dark","");
		} else {
			this._urlState.searchParams.delete(".dark");
		}
		this._urlState.hash = "";
		
		this.showScreen();
	}
}
