/* Abelbeck Aviation Checklist - asyncio library script
Copyright 2024 Frank Abelbeck <frank@abelbeck.info>

This file is part of the Abelbeck Aviation Checklist (AAC) toolbox.

The AAC toolbox is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

The AAC toolbox is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the AAC toolbox.  If not, see <http://www.gnu.org/licenses/>.
*/


export class AsyncIOBasic {
	constructor() {
		const methods = Object.getOwnPropertyNames(Object.getPrototypeOf(this));
		methods.filter((method) => (method !== "constructor")).forEach((method) => { this[method] = this[method].bind(this);});
	}
	
	static async sleep(intMilliseconds) {
		await new Promise(resolve => setTimeout(resolve, intMilliseconds));
	}
}


export class AsyncIOLock extends AsyncIOBasic {
	constructor() { super();
		this._strLock = crypto.randomUUID();
	}
	
	async request(strMode,fnAsyncCriticalPath,fnAsyncOnError) {
		let errError;
		if (typeof(fnAsyncCriticalPath) === "function") {
			try {
				await navigator.locks.request(
					this._strLock, 
					{ mode:strMode },
					async (lock) => {
						try {
							await fnAsyncCriticalPath();
						} catch (err) {
							errError = err;
							try {
								await fnAsyncOnError(err);
							} catch (err2) {}
						}
					}
				);
			} catch (err) {
				errError = new Error(`could not obtain read lock: ${err}`);
			}
		}
		if (errError) {
			throw errError;
		}
	}
	
	async read(fnAsyncCriticalPath,fnAsyncOnError) {
		await this.request("shared",fnAsyncCriticalPath,fnAsyncOnError);
	}
	
	async write(fnAsyncCriticalPath,fnAsyncOnError) {
		await this.request("exclusive",fnAsyncCriticalPath,fnAsyncOnError);
	}
}

export class AsyncIOMapSet extends AsyncIOBasic {
	constructor(varInit) { super();
		if (varInit &&
			typeof(varInit.has) === "function" &&
			typeof(varInit.delete) === "function" &&
			typeof(varInit.clear) === "function") {
			// mapInit seems to implement a basic map or set interface: use as var object
			this._var = varInit;
		} else {
			// create a new map object
			this._var = undefined;
		}
		this._lock = new AsyncIOLock();
	}
	
	async has(strKey) {
		let boolHas = false;
		await this._lock.read(async () => {
			boolHas = this._var.has(strKey);
		})
		return boolHas;
	}
	
	async delete(strKey) {
		await this._lock.write(async () => {
			this._var.delete(strKey);
		});
	}
	
	async clear() {
		await this._lock.write(async () => {
			this._var.clear();
		});
	}
	
	async forEach(fnAsyncCriticalPath) {
		await this._lock.read(async () => {
			this._var.forEach( async (value,key) => {
				await fnAsyncCriticalPath(key,value);
			});
		});
	}
}


export class AsyncIOMap extends AsyncIOMapSet {
	constructor(mapInit,fnSet,fnGet) { super(mapInit);
		if (mapInit &&
			typeof(mapInit.get) === "function" &&
			typeof(mapInit.set) === "function") {
			// mapInit seems to implement a map interface: use as map object
			this._var = mapInit;
		} else {
			// create a new map object
			this._var = new Map();
		}
	}
	
	async get(strKey) {
		let value;
		await this._lock.read(async () => {
			value = this._var.get(strKey);
		});
		return value;
	}
	
	async set(strKey,value) {
		await this._lock.write(async () => {
			this._var.set(strKey,value);
		});
	}
}


export class AsyncIOSet extends AsyncIOMapSet {
	constructor(setInit) { super(setInit);
		if (setInit &&
			typeof(setInit.add) === "function") {
			// mapInit seems to implement a set interface: use as set object
			this._var = setInit;
		} else {
			// create a new map object
			this._var = new Set();
		}
	}
	
	async add(strKey) {
		await this._lock.write(async () => {
			this._var.add(strKey);
		});
	}
	
	async union(other,boolInPlace=true) {
		let setRet;
		await this._lock.write(async () => {
			setRet = this._var.union(other);
			if (boolInPlace) this._var = setRet;
		});
		return setRet;
	}
	
	async difference(other) {
		let setRet;
		await this._lock.write(async () => {
			setRet = this._var.difference(other);
			if (boolInPlace) this._var = setRet;
		});
		return setRet;
	}
	
	async intersection(other) {
		let setRet;
		await this._lock.write(async () => {
			setRet = this._var.intersection(other);
			if (boolInPlace) this._var = setRet;
		});
		return setRet;
	}
	
	async symmetricDifference(other) {
		let setRet;
		await this._lock.write(async () => {
			setRet = this._var.symmetricDifference(other);
			if (boolInPlace) this._var = setRet;
		});
		return setRet;
	}
	
	async isDisjointFrom(other) {
		let boolRet = false;
		await this._lock.write(async () => {
			boolRet = this._var.isDisjointFrom(other);
		});
		return boolRet;
	}
	
	async isSubsetOf(other) {
		let boolRet = false;
		await this._lock.write(async () => {
			boolRet = this._var.isSubsetOf(other);
		});
		return boolRet;
	}
	
	async isSupersetOf(other) {
		let boolRet = false;
		await this._lock.write(async () => {
			boolRet = this._var.isSupersetOf(other);
		});
		return boolRet;
	}
}
